// MASTER.H 1.25 - Header file with definitions for MSC program
// 1.01: 5-21-93: Added enum & table entries for Interior Temperature
// 1.03: moved Cint, cmd_t from page, added param_t
// 1.04: redefine getwix()
/* 1.1:  added record::set_ts(ts),set_ts(void),move(record &)
	 added endprog = 21 for out-of-range record/field ints */
// 1.2   added fixstream()
// 1.22  variable length rlists
// 1.251:corrected parameter limit

#ifndef MASTERH
#define MASTERH

extern "C" {
#include "..\inc\kernel.h"
#include "..\inc\system.h"
};
#include "..\inc\msclib.h"

#include <stdlib.h>
#include <iostream.h>
#include <time.h>
#include <fstream.h>
#include <iomanip.h>
#include <strstrea.h>
#include <dos.h>
#include <bios.h>
#include <string.h>
#include <conio.h>
#include <ctype.h>
#include <dir.h>

// #define TRACE 1
#define EDOK DOSERR_SUCCESS
const static char *msc_version = "1.251";
const static unsigned SHER_WAIT = 5000;
const static int S_LEN = 512;   // stack size in ints (1KB)
const static USHORT H1PRI = THREAD_PRIORITY_DEFAULT + 1;
const static USHORT NPRI = THREAD_PRIORITY_DEFAULT;
const static USHORT L1PRI = THREAD_PRIORITY_DEFAULT - 1;
const static PRINTLOG = 0; // logs are printed if > 1, msgs if > 0
const static char day_in_month[] = { 31,28,31,30,31,30,31,31,30,31,30,};

// record parameters for strings
const static int MAX_CHAR = 550;  // data and extras
const static int NUM_USE = 10;     // number of execution flags
const static char SEP = ',';
const static char  *NL = "\r\n";	  // newline - definable

// record parameters for flag ( values 0 to 7 are reserved for retry count)
// values of 0x10 to 0x17 signify that the record should be resent
const int MAX_CCF_RETRY = 5;       // max 7
const int N_FLAG = 0x18;	   // record is waiting for processing
const int D_FLAG = 0x19;	   // record has been processed
const int Z_FLAG = 0x1a;	   // invalid record
const int E_FLAG = 0x1c;	   // retries have expired
const int R_FLAG = 0x1e;	   // NAK received, retry first
const int RETRY_FLAG = 0x1f;       // Get either a retry or a N_FLAG
const int C_FLAG = 0x20;	   // failed checksum
const int I_FLAG = 0x21;           // invalid flag

// rlist parameters
const int MAX_POINTERS = 450;
const int MIN_DUMP = 50;
const int PATH_LENGTH = 32;
static const char path[] = {"C:\\MSC\\DATA"};   // path for data files;1.03 moved from lists
const int MBD_ONLY = 0x10;  // only store MBC data in del_rec()
const int NOT_DONE_ONLY = 0x20;  // only store data not done in del_rec

// used for debug purposes
#define MAX_TRACE 22

const static int STKMARK = 1234;
const static int MIN_MSGS = 220;   // minimum free message spaces
const static int MIN_DISK_FREE = 10; // 1.02 was 1

// index to local_op_mode:
const static int DISP_MODE = 0; // see below
const static int NEW_PAGE = 1;  // operator changed the page
// const static int NEW_DATA = 2;  // page needs updating

// display modes for local_op_mode[0]:
const static int LOG_PAGE = 60; // F2
const static int STAT_PAGE = 61; // F3
const static int NUM_DISP = 45; // number of items to update for new page
// extended key codes:
const static int PG_UP = 73;
const static int PG_DN = 81;
const static int HOME = 71;
const static int END = 79;
const static int UP_AROW = 72;
const static int DN_AROW = 80;

// compiler requires unique name as between all enumerations

// units are functional objects which may have no program implementation
enum unit { invunit, archive, bite, events, logger, mbc, ccf, pri, mult,
		program, printer, parser, analog, backup, digital, control,
		msgdata, pwranal, weather, };

typedef struct  {		  // defines object properties
unit u;			// unit enumeration
char i[3];		// string to put into records and messages
char a[13];		// string to display
int addr;   	// I/O address
unsigned id;	// port ident
int      row;   // row to display communications statistics
} unitint;

/* message types are used for two purposes:
   * identify the CCF/MSC message data/command format
   * identify an rlist (unique to an rlist for archiving & debug reasons) */
enum msgtype { invtype, bitedata, com_stat, deny, log_msg, mbc_cmd, mbc_msg,
	   arc_cmd, arc_data, pgm_ctl, ro_cmd, ro_data, stat_msg,
	   sno_data, retry, envdata, ms_cmd, confirm, mbc_data, };

typedef struct {	// provides debug and message strings
msgtype t;	    // msgtype enumeration
char i[4];	    // code for both messages and subdirectory id
char a[13];	    // string for debug messages
int  row;           // row to display rlist statistics
int recs;           // number of record pointers
} typeint;

// "outside world" inputs - used for defining hardware parameters
enum sensor {
	door,		// door open switch
	air,		// A/C air flow switch
	motion,		// Motion detector
	fsmoke,		// Front Smoke detector
	bypass,		// UPS is in bypass mode
	pwr_loss,	// UPS senses no utility power
	rsmoke,		// Rear Smoke Detector
	pri_main,	// Control voltage to Primary PA main relay
	pri_stby,	// Control voltage to Primary PA standby relay
	back_main,	// Control voltage to Backup PA main relay
	back_stby,	// Control voltage to Backup PA standby relay
	fwd_pwr,	// Forward RF power
	ref_pwr,	// Reflected PA power
	level,		// Auto level control output voltage
	latch,		// PA operate relay latched
	unreg,		// Control unit supply voltage
	fv_supply,	// +5 V supply to STA-U #1
	ADC_mon,	// Converter test point - 0xco
	PA_sel,		// Control voltage to PA coax relays
	pwr_mon,	// Control voltage to Facility power relay
	pwr_fdbk,	// Aux contacts of Facility power relay
	intemp,         // Interior temperature
	invsens,	// invalid
};

typedef struct {		// hardware parameters
	sensor s;			// sensor enumeration
	char i[3];			// id code
	char a[17];			// debug message string
	unsigned KPort:3;	// 0=PA,1=PB,2=PC,3=DAS-4(analog),4=DAS-4(digital),5 invalid
	unsigned KBit:3;	// bit of PIO-24 or DAS-4 mux or bit (Backup MSC)
	unsigned KInv:1;	// 0 = non-inverted, 1 = inverted
	unsigned LptSelect:3; // value of bit 7 of lpt output for multiplexer
	unsigned LptBit:3;	// bit in lpt1 status reg/analog scale factor
	unsigned LptInv:1;	// 0 = non-inverted, 1 = inverted
	int      row;
	int      col;
	} sensorint;

void far ChkWatchDogResponder(void);
void far PgmWatchDogResponder(void);
void far disp_use(void); // used in WatchDogs to show activity
void log(unit, const char*, int, int=1); // puts a new record into loglist rlist
void log(msgtype, char*, int, int=1);
int main(int, char **);
int getix(const unit);			// returns index in Uint or neg
int getix(const msgtype);
int getix(const sensor);

// the following return the length of the string, name is a, number is i
unit GetUnit(char *);  // param is the 2-letter code Uint.i
int GetUnitName(char *,const unit);	 // returns index, puts into Uint.a
int GetUnitCode(char *,const unit);	 // returns index, puts into Uint.i
msgtype GetType(char *); // param is the 3-letter code Tint.i
int GetTypeName(char *,const msgtype);  // returns index, puts into Tint.a
int GetTypeCode(char *,const msgtype);  // returns index, puts into Tint.i
sensor GetSensor(char *); // param is the 2-letter code
int GetSensorName(char *,const sensor); // returns index, puts into Sint.a
int GetSensorCode(char *,const sensor); // returns index, puts into Sint.i

// the following overload the insert ( << ) operator for the enums:
ostream &operator<<(ostream&, unit);	// writes the id code
ostream &operator<<(ostream&, msgtype); // writes the id code
ostream &operator<<(ostream&, sensor);  // writes the id code
istream &operator>>(istream&, msgtype&); // gets the type from id code,needs STX

// Weather Monitor commands
enum wcmd{ istart,// send n(0-63) images (0=65536)
	istop,			// stop sending images
	set_in_low,     // set low reporting limit for internal temperature
	set_in_high,    // set high reporting limit for internal temperature
	set_out_low,    // set low reporting limit for outside temperature
	set_out_high,   // set high reporting limit for outside temperature
	set_baro_low,   // set low reporting limit for barometric pressure
	set_baro_high,  // set high reporting limit for barometric pressure
	set_hum_low,    // set low reporting limit for internal humidity
	set_hum_high,   // set high reporting limit for internal humidity
	set_wind_high,  // set high reporting limit for wind speed
	set_warchive,	// set archive period to n(0-63) minutes,0=def
	set_wsample,	// set sample period to n(0-63) seconds,0=def
	set_whour,		// set clock hour to n(0-23)
	set_wmin,		// set clock minutes to n(0-59)					
	set_wsec,		// set clock seconds to 0						 
	set_wday,		// set day of month to n(1-31)
	set_wmonth,		// set month of year to n(1-12)				   
	get_wtime,		// get time
	get_wdate,		// get date
	get_hiws, 		// get high wind speed & time
	get_hlit, 		// get high & low inside temperature
	get_hlot, 		// get high & low outside temperature
	get_hlhum,		// get high & low humidity
    clr_hiws,       // clear high wind speed
    clr_loit,       // clear low inside temperature
    clr_loot,       // clear low outside temperature
    clr_lohum,      // clear low humidity
    clr_hiit,       // clear high inside temperature
    clr_hiot,       // clear high outside temperature
    clr_hihum,      // clear high humidity
	aenable,		// enable archive timer
	adisable,		// disable archive timer
	ustart,			// start updating & archiving
	ustop,			// stop updating & archiving
	get_psample,	// get sample period
	get_parchive,	// get archive period
	get_image,		// force sample of the image
	invwcmd,	  };


typedef struct {
	wcmd	w;  // the command inentifier
	char i[4];  // identifier for messages
	int	    c;  // numerical command from CCF
	int	    m;  // range of parametric CCF commands for this command
	char s[6];  // the string part of the weather monitor command
	int	    b;  // format flag:size in bytes of parameter,fixed
	long	d;  // up to 3 fixed command bytes, send ls first
	int	    l;  // number of bytes to receive
	} weatherint;

// Weather Monitor output for the istart command
typedef struct {
	int  pos;	// index in recbuf of first(LS) byte
	int  len;	// no of bytes (Intel order)
	int  low;	// lower limit
	int high;	// upper limit
	int	d;		// no of digits to put in the message
	} im_type;

const static int NUM_UNIT=19;

const static unitint Uint[] = {
{     ccf, "CC" , "Portland"	,    0,  4, 22 },
{     pri, "PM" , "Main Line"   ,0x2B0,  2, 11 },
{     mbc, "MB" , "MBC"		,0x2A8,  1, 15 },
{    bite, "BT" , "testing"	,    0,  0, 22 },
{    mult, "MT" , "multiple"	,    0, 94, 22 },
{  analog, "AN" , "A/D board"   ,0x2C0, 10, 22 },
{  backup, "BM" , "Backup Line" ,0x2A0,  0, 13 },
{ digital, "PI" , "PIO24 board" ,0x2D0, 20, 22 },
{  events, "EV" , "Alarm msgs"  ,    0, 85, 22 },
{  logger, "LG" , "logging"	,    0,  0, 22 },
{  parser, "PS" , "cmd parser"  ,    0, 93, 22 },
{ archive, "AR" , "archive"	,    0, 96, 22 },
{ control, "CT" , "PA Control"  ,    0, 80, 22 },
{ invunit, "IV" , "invalid"	,    0, 99, 23 },
{ msgdata, "DA" , "messages"	,    0, 98, 22 },
{ printer, "PR" , "printer"	,0x378,  1, 22 },
{ program, "MS" , "MSC program" ,    0, 95, 22 },
{ pwranal, "PA" , "Power Anal." ,0x2B8,  3, 16 },
{ weather, "WM" , "Serial Env." ,0x2B8,  3, 17 }, };

const static int NUM_TYPE = 19;

const static typeint Tint[] = {
{  confirm, "ACK" ,"acknowlegmt", 22,   0},
{ bitedata, "BTD" ,"test data",   22,   0},
{ com_stat, "CST" ,"comm. stats", 22,   0},
{   ms_cmd, "EVC" ,"CCF command", 11, 100},
{  envdata, "EVD" ,"Environment", 17, 100},
{  invtype, "IVT" ,"inv. type",	  24,   0},
{  log_msg, "LOG" ,"log message", 16, 100},
{  mbc_cmd, "MBC" ,"MBC command", 13, 450},
{ mbc_data, "MBD" ,"MBC data",	  14, 100},
{  arc_cmd, "MRC" ,"archve cmd.", 22,   0},
{ arc_data, "MRD" ,"archive data",22,   0},
{  pgm_ctl, "MSC" ,"program ctl", 22,   0},
{  mbc_msg, "MSD" ,"Msgs. to CCF",12, 450},
{ stat_msg, "MSS" ,"Sensor Data", 15, 100},
{   ro_cmd, "ROC" ,"rem. op. cmd",19, 100},
{  ro_data, "ROD" ,"rem.op. data",18, 100},
{    retry, "RTY" ,"retry msg",	  22,   0},
{     deny, "NAK" ,"unknown cmd.",23,   0},
{ sno_data, "SEN" ,"sensor data", 22,   0}, };

const static int NUM_SENS = 22;

const static sensorint Sint[] = {
{ door	, "DS","door switch"	      ,0,0,1,0,7,0,3,1},
{ air	      , "AF","A/C filter"  	      ,0,1,1,0,6,1,3,14},
{ motion    , "MD","motion det."	      ,0,2,0,0,5,0,3,25},
{ fsmoke    , "FS","front smoke det." 	,0,3,0,0,3,0,4,1},
{ bypass    , "BY","UPS inverter on" 	,0,4,0,1,7,1,4,18},
{ pwr_loss  , "PL","UPS  alarm"      	,0,5,1,1,6,1,5,18},
{ rsmoke    , "RS","rear smoke det."	,0,6,0,1,5,0,5,1},
{ pri_stby  , "PS","Pri stby rly in" 	,2,1,0,2,0,0,3,55},
{ back_stby , "BS","Back stby rly in"	,2,3,0,2,0,0,3,55},
{ pri_main  , "PM","Pri main rly in" 	,2,0,0,2,0,0,3,55},
{ back_main , "BM","Back main rly in"	,2,2,0,2,0,0,3,55},
{ fwd_pwr   , "FP","Fwd PA power"	,3,0,0,3,1,0,4,64},
{ ref_pwr   , "RF","Ref. PA power"   	,3,1,0,3,1,0,5,65},
{ level	    , "CV","PA control volt" 	,3,2,0,3,2,0,6,64},
{ latch	    , "FL","PA op. confirm"  	,3,3,0,3,4,0,7,64},
{ intemp    , "TE","Int. Temperature"   ,3,4,0,3,1,0,6,33},
{ unreg	    , "UV","Line Voltage"    	,3,5,0,3,6,0,6,17},
{ fv_supply , "FV","+5 V control"       ,3,6,0,3,2,0,7,17},
{ ADC_mon   , "TV","Test Voltage"	      ,3,7,0,3,4,0,7,22},
{ PA_sel    , "CX","PA coax rly in"	      ,4,4,0,4,0,0,3,40},
{ pwr_fdbk  , "PR","Line Disconnect"      ,4,5,1,5,0,1,2,14},
{ pwr_mon   , "LV","Utility Off" 	      ,4,6,0,5,0,0,2,1},
{ invsens   , "IS","invalid"		      ,5,0,0,6,0,0,24,70},
};

const static int NUM_WTHR = 39;

const static weatherint Wint[] = {
{istart			,"IMG",1020,0x40, "LOOP",-22,   0x10000, 18,},
{istop			,"IST",1000,   1,  "RRD",  3,  0x014200,  1,},
{set_in_low     ,""   ,1440,  60,  ""   ,  0,    125670, 10,},
{set_in_high    ,""   ,1500, 130,  ""   ,  0,    792381, 10,},
{set_out_low    ,""   ,1630,  60,  ""   ,  1,    125670, 10,},
{set_out_high   ,""   ,1690, 130,  ""   ,  1,    792381, 10,},
{set_baro_low   ,""   ,1820, 340,  ""   ,  4,    125670,100,},
{set_baro_high  ,""   ,2160, 340,  ""   ,  4,    792381,100,},
{set_hum_low    ,""   ,2500, 100,  ""   ,  5,    125670,  1,},
{set_hum_high   ,""   ,2600, 100,  ""   ,  5,    792381,  1,},
{set_wind_high  ,""   ,2700, 200,  ""   ,  2,    792381,  1,},
{set_warchive	,"SSP",1100,0x40,  "SSP",-11,     0x100,  0,},
{set_wsample 	,"SAP",1200,0x40,  "SAP",  1,	      0,  0,},
{set_whour   	,"SHR",1370,0x20,  "WWR", 52,    0xbe23,  0,},
{set_wmin		,"SMN",1300,0x40,  "WWR", 52,    0xc023,  0,},
{set_wsec		,"SSC",1270,   1,  "WWR", 52,    0xc223,  0,},
{set_wday		,"SDY",1400,0x20,  "WWR", 52,    0xc823,  0,},
{set_wmonth  	,"SMN",1420,0x10,  "WWR", 12,    0xca23,  0,},
{get_wtime   	,"GTM",1001,   1,  "WRD",  2,    0xbe64,  3,},
{get_wdate   	,"GDT",1002,   1,  "WRD",  2,    0xc844,  2,},
{get_hiws    	,"GWS",1010,   1,  "WRD",  2,    0x6082,  4,},
{get_hlit    	,"GIT",1011,   1,  "WRD",  2,    0x3484,  4,},
{get_hlot    	,"GOT",1012,   1,  "WRD",  2,    0x5a84,  4,},
{get_hlhum   	,"GHU",1013,   1,  "WRD",  2,    0x8244,  2,},
{aenable	 	,"EBT",1003,   1,  "EBT",  0,         0,  0,},
{adisable		,"DBT",1004,   1,  "DBT",  0,	      0,  0,},
{ustart	  		,"SRT",1005,   1,"START",  0,         0,  0,},
{ustop	   		,"STP",1006,   1, "STOP",  0,	      0,  0,},
{get_psample 	,"GPS",1007,   1,  "RRD",  3,  0x013a01,  1,},
{get_parchive	,"GPA",1008,   1,  "RRD",  3,  0x013c01,  1,},
{get_image   	,"GIM",1009,   1,  "IMG",  3,  0x171c01, 12,},
{clr_hiws       ,""   ,1090,   1,  "WWR",  3,  0x006021,  0,},
{clr_loit       ,""   ,1091,   1,  "WWR",  4,0xff7f3843,  0,},
{clr_loot       ,""   ,1092,   1,  "WWR",  4,0xff7f5e43,  0,},
{clr_lohum      ,""   ,1093,   1,  "WWR",  3,  0x7f8623,  0,},
{clr_hiit       ,""   ,1094,   1,  "WWR",  4,0x00803443,  0,},
{clr_hiot       ,""   ,1095,   1,  "WWR",  4,0x00805a43,  0,},
{clr_hihum      ,""   ,1096,   1,  "WWR",  4,  0x808223,  0,},
{invwcmd	 	,"IVC",2900, 100,	  "",  0,         0,  0,},
};

const static NUM_PWR = 3; // number of Sherlock commands

typedef struct {
	int   cmd;  // the CCF command parameter
	char s[7];  // the keypresses needed
	int	 n;  	// the number of keypresses
	} sherlockint;

const static sherlockint Pint[] = {
	{ 3000, "OMDSUS", 6, }, // get summary record
	{ 3001,"OMDSDDS", 7, }, // get all events record
	{ 3002,"OMUSDSS", 7, }, // clear event counter
};

class field { // holds a variable-length string of chars
protected:
	char far *s;  // null-terminated!
	int len;	  // length not including null

public:
	field(void) {len=0; s = 0;};  // constructor for an empty string
	field(const char *);		   /* constructor for initializing with
					  an array of characters			  */
	field(int size, const char *); // truncated initializer
	field(int size, char); /* constructor for initializing with
					a fixed array of preset characters	*/
	field(const field &);		 // copy initializer
	~field(void); // destructor
	int getfs(int,char *) const; // copies the string, returns length
	int getn(void) const;	// returns the length of the string
	field &operator=(const field &);   // overloads assignment operator
	void trim(void);				// removes trailing white space
	friend field &operator+(const field &, const field &); // catenation
	friend int operator==(const field &, const field &); /* True(1) if LHS
				is identical to RHS except for trailing blanks, */
	friend int operator!=(const field &, const field &); // True if not same
	friend int operator>(const field &, const field &); /* True if LHS is
			above (higher value) RHS in ASCII collating order */
	friend int operator>=(const field &, const field &); /* True if LHS is
						 identical to RHS or above in ASCII collating order */
	friend int operator<(const field &, const field &); /* True if LHS is
			 below RHS in ASCII collating order */
	friend int operator<=(const field &, const field &); /* True if LHS is
						 identical to RHS or below in ASCII collating order */
	int operator&(const field &); // True if string is contained in s
	char operator[](int) const;  // must be member function, returns char
	friend ostream &operator<<(ostream &,const field &); // overloads insertion
};

const time_t min_year=0x2a000000; // April '92
const time_t max_year=0x5e000000; // Jan 2020

class ts {				 // timestamp
	time_t secs;

	int cv_an_int(char c) const {
		if(c > '9') return(toupper((int)c) - 55);
		else return ((int)c - 48);
		}

	char cv_int_an(int i) const {
		if(i > 9) return (char)(i + 55);
        	else return (char)(i + 48);
        	}

        struct tm *gettm(void) { return localtime(&secs); };

public:
	ts(void) { secs = time(0);};   // constructor for now
	ts(const time_t itt) { secs = itt; };
	ts(const ts &t1) { secs = t1.secs; };	// copy initializer
	ts(const char *); // makes a ts from a hex string (0 if invalid)
	~ts(void) { secs = 0L; };		// destructor
	void now(void) { secs = time(0); }; // resets time to now
	int getday(void) { return gettm()->tm_mday; };
	int getmon(void) { return gettm()->tm_mon + 1; };
	int getyr(void) { return (gettm()->tm_year + 1900); };
	int getsec(void) { return gettm()->tm_sec; };
	int getmin(void) { return gettm()->tm_min; };
	int gethr(void) { return gettm()->tm_hour; }; 
	friend time_t operator-(const ts &, const ts &); /* returns
		difference in secs */
	void addsec(const time_t &s) { secs += s;};  // adds seconds to timestamp
	int getts(char *) const;	 // converts to text string, returns length
	ts &operator=(const ts &);	 // overloads assignment operator
	int getfn(char *) const; // makes a file name in a field from ts
	int valid(void) const { return (secs > min_year && secs < max_year)?1:0;};
	friend ostream &operator<<(ostream &,const ts &); // overloads insertion
};


class record:public field { // holds a delimited string of chars
protected:
	ts  created;		  // timestamp
	int flag; // record valid
	int seqno;	 // for retry ident
	int nf;   // number of lines for display
public:
	record(void) { nf = 0; seqno = -1; flag = Z_FLAG; };
	record(const ts &, int, const char *, ...);
	record(const char *, ...);  // use now for timestamp, last is NULL
	record(const int, const char * ...); // first param is no of strings
	record(const ts &t1, const field &f1) :	field(f1) { created = t1;
		nf = 1; flag = N_FLAG; };
	record(const field &f1):field(f1) { nf = 1; flag = N_FLAG; }; // does not count delimiters
	record(const record &r1); // copy initializer
	~record();
	void move(record &); // eliminates source record data, copies pointer
	void set_nf(int i) { nf = i; };
	int get_nf(void) const; // returns the number of extra lines
	void set_flag(int); // also increments retry count for RETRY_FLAG
	// if flag is N_FLAG, sets 0, else increments up to limit
	int get_flag(void);
	void set_seqno(int seq) { seqno = seq; };
	int get_seqno(void) { return seqno; };
	int add_str(const char *, char sep_char = SEP);  // add to end of string,returns no of SEP
//	void del_fld(int);		 // deletes section at index
	int get_fld(int,char *);  // returns selected section and length
	int add_fld(const field &);  // add a field to existing record
	void add_ts(time_t t1) { created.addsec(t1); }; // test only
	void set_ts(ts t1) { created = t1; }; // copies the timestamp
	void set_ts(void) { created.now(); }; // resets timestamp to current
	ts &get_ts(void) { return created; };
	record &operator=(const record &); // = overload
	int get_str(int, char *); // converts a record to a string,
	//  including encoded number of sections and timestamp
};


class rlist {
	ts origin;			// time rlist was created
	msgtype type;                   // type of record contained
	int max_rec;                    // number of record pointers
	int f;				// flag: N_FLAG, except E_FLAG for fixed
	record  *list[MAX_POINTERS];	// An array of pointers to records
	int nrec;		// The number of records
	char fname[PATH_LENGTH];   // filename for disk operations
	int plen;				  // position to put filename body
	int windx;	   // next record in wptr to write, -1 if none open
	int next,last;    // indices of records to display at top of screen
	int set_last(int); // gives number of records to display
	int stat_change;     // nonzero if new stats to display
	fstream diskstrm;    // for dosk operations
public:
	rlist(const msgtype);    // type, number of entries
	rlist(const rlist &);		 // copy initializer
	rlist(void);				  // should have one
	~rlist();
	int add_rec(const record &);  // sorted by timestamp, returns nrec
	int add_rec(const char *);	// may include # of sections & timestamp
	int copy_rec(const rlist&, int); // copies to end with now time
	int del_rec(int=0);  // archives processed entries,returns no transferred
	int find_trec(const ts &); // find index of nearest record
	int find_rec(int) const;		// find first record with specified flag
	int find_seq(int) const; // get lowest record with sq1==seqno
	int get_nrec(void) const { return nrec; };
	int get_mrec(void) const { return max_rec; };
	void set_next(int);
	ts &get_ts(void) { return origin; };
	int get_f(void) { return f; };
	msgtype get_msgtype(void) { return type; };
	rlist &operator=(const rlist &);  // = overload
	record &operator[](int); // returns ref to indexed record
	friend int operator==(const rlist &, const rlist &); // true if hdrs equal
	int empty(void);
	int wopen(int time_rec = MAX_POINTERS+1); // opens a file for writing,-="DUMP"
	int ropen(const ts&);		   // opens a file for reading
	int wnext(int last_rec = MAX_POINTERS+1); // writes one record,-=only N_FLAG
	int rnext(void);		 // adds one record from disk
	int load(const char *);  // loads a file of sample data
	void diskclose(int);	// closes a file, optional number of records written
	friend ostream &operator<<(ostream &,const rlist &); // reads statistics
	void display_stats(void);   // displays rlist statistics
	void display_page(void); // displays one screenfull at next
};


class queue {
	int *entries;
	int indx;
	int max_entries;
public:
	queue(int);
	~queue(void);
	int is_empty(void) { return (indx < 0) ? 1 : 0; };
	int is_full(void) { return (indx == max_entries - 1) ? 1 : 0; };
	int has_data(void) { return (indx >= 0) ? 1 : 0; };
	int push(int); // returns -1 if full
	int pop(void); // returns -10000 if empty
	int load(const char*); // loads with numbers from file in path, returns number of entries
	void clear(void) { indx = -1; };
	};

// parameters for com_port status word
const int INSTALLED   = 1;
const int IRQSET	  = 2;
const int UART_INIT   = 4;
const static int PORT_SET	= INSTALLED | IRQSET | UART_INIT;
const int C_MASK	  = 8;	  // new line handshake values
const int H_MASK	  = 0x10;  // CTS, DST and CD all set
const int STATE	   = 0x60;  // bits 5 & 6
		 /*  0 = idle, waiting for ENQ or data to send (no timeout)
			 0x20 = getting message from MBC, waiting for EOT
			 0x40 = ENQ sent, waiting for ACK
			 0x60 = message sent to MBC, waiting for ACK or NAK */
const int BAD_MSG	 		 = 0x80;	// set on bad message received
const int REC_LINE_AV 		 = 0x100;	// set on full line received, cleared on use
const int BUF_OVERRUN 		 = 0x200;	// on receive
const int RECEIVE_ERROR 	 = 0x400;	// parity or framing error
const int MISSED_LINE 		 = 0x800;	// on transmit
const int BREAK_DETECT 		 = 0x1000;	// same as PCCOM word bit 12
const static int TX_BUF_BUSY = 0x2000;	// cleared when full line sent
const int TX_IDLE	 		 = 0x4000;	// same as PCCOM word bit 14

// parameters for PCCOM
const int COMIRQ=4; // common for all ports
const int MCR=3;		//  sets DTR & RTS,OUT1(bit 2)&OUT2(bit 3)=0
const int VECTOR=0x02BF; 	// pccom sets this location on interrupt
const int ACTIVE=0;	  		// active low: vector is 0 for interrupt
#define NO_X 0
#define X_FLOW 1

// fixed parameters for UART
const unsigned int STOPBITS   = 1;

// parameters for line status word from PCCOM
const int CTS_CHANGE  = 1; // clear to send - pin 4
const int DSR_CHANGE  = 2; // data set ready - pin 6
const int RING_TRAIL  = 4;
const int CD_CHANGE   = 8; // carrier detect - pin 8
const int LINE_CHANGE = CTS_CHANGE | DSR_CHANGE | CD_CHANGE; // 1.23 no ring trail
const int CTS		  = 0x10;
const int DSR		  = 0x20;
const int RING		  = 0x40;
const int CD		  = 0x80;
const int DATA_READY  = 0x100;
const int OVERRUN_ERR = 0x200;
const int PARITY_ERR  = 0x400;
const int FRAMING_ERR = 0x800;
const int HANDSHAKE   = CTS | DSR | CD;
const int THRE		  = 0x2000; // Transmitter Holding Register Empty

// parameters for line protocols
const static char SOH=1;
const static char STX=2;
const static char TAB=9;
const static char ETX=3;
const static char EOT=4;
const static char ENQ=5;
const static char ACK=6;
const static char  LF=0x0A;
const static char  CR=0x0D;
const static char NAK=0x15;
const static char SYN=0x16;
const static char ESC=0x1b;
const static char  FS=0x1C;

// communications statistics
const int SEND_CNT  = 0;
const int REC_CNT   = 1;
const int ERROR_CNT = 2;

// MBC - MSC interface protocol parameters
const int MAX_RETRY = 5;  // for sending to MSC and Weather Monitor
const int MBC_TIMEOUT = 5000;   // (ms) No reply to ENQ or message
const int MSC_TIMEOUT = 4000;   // (ms) No reply to ACK (after MSC ENQ)

class port {   // variables and methods common to all address-register I/O
protected:
	int ident;  // as used in bios calls
	unsigned addr;  // port address
	unit u;
	int status;  /* bitfield: for comports:
		0x01 	1 = installed, 0x02 vector set, 0x04 baud, etc set,
		0x08 	1 = line change (1 = changed)
		0x10 	1 = handshake OK (CTS, DSR, CD, Xon/off)
		0x20,0x40 = protocol State (for MBC interface)
		0x80 	1 = received bad message
		0x100 	1 = received record available,
		0x200 	1 = overrun of receive ring buffer,
		0x400 	1 = receive error
		0x800 	1 = transmit missed line,
		0x1000 	1 = break on receive
		0x2000 	1 = transmit buffer not empty,
		0x4000 	1 = transmit idle.  */

			/* bitfield for sensor ports:
		0x01 	1 = board installed
		0x02 	1 = test passed	
		0x03 	1 = data ready to output
		0x04 	1 = data has changed
		0xf0   	  = number of sensors read */

public:
	port(const unit u1);
	~port();
	port(void);		// default constructor: do not use
	port(const port &); // copy constructor: do not use
	port &operator=(const port &); // = overload:do not use
	int getpid(void) { return ident; };
	void setpstatus(int, int); // data, mask
	int getpstatus(int mask = 0x7fff) { return (status & mask); };
	int getpaddr(void) { return (int)addr; };
	unit getpunit(void) { return u; };
	};

class com_port:public port {  // variable and methods common to all serial ports
protected:
	unsigned rate;	// baud rate, 300 min, 19200 max, multiples of 150
	unsigned bits;	// 7 or 8 allowed only
	unsigned parity;  // 0 = no parity, 1 = odd parity, 3 = even parity
	unsigned flow;   // XON = 1 = enabled, 0 = XOFF = disabled
	int num_send;	 // 0 = send string, else number of binary bytes
	int send_indx;			   // current record being sent
	static int IRQset;	 // set to IRQ number at first installation
	char *recbuf; // to get incoming lines, make room for SOH
	char *sendstr;	// line to send, room also for checksum
	int sindx, rindx;	// indices for sendstr & recbuf
	int msg_cnt[3];	  // sent, received, errors
	int Transmit(void);  // sends all available characters,sets sindx to -1
	int GetMsgCnt(int);
	int stat_change;     // non-zero if msg_cnt changes
public:
	com_port(void);
	com_port(unit u1);
	~com_port();
	com_port(const com_port&); // copy initializer
	com_port &operator=(const com_port &); // = overload
	int ComInitPort(int r, int b, int p, int f);  // set up UART
	int CheckStatus(void);  /* Checks line status, sets status,
		returns pos number of available input, or -1 */
	void add_check(void); /* uses sendstr & sindx:calculates,adds,
	  & returns the checksum */
	int get_check(void); /* uses recbuf & rindx. returns -ve if bad,
	  strips check field, param is 1 for CRC */
	friend ostream &operator<<(ostream &, com_port &); // sends statistics
	void display_stats(void); // row to display on
	};

void p_reset(void); // MBC timer handler

class mbc_port:public com_port { // MBC port
	int active;  // if non-zero, may transmit to the MBC
public:
	void set_active(int s) { active = s; stat_change = 1;};
	int get_active(void) { return active; };
	int p_timed_out;	  // defines the state of the protocol timer
	mbc_port(unit u1);
	mbc_port(const mbc_port &);
	mbc_port(void);
	~mbc_port();
	mbc_port &operator=(const mbc_port &);
	void reset(void);
	int CheckEnd(int);	 // calls CheckStatus, sets REC_LINE_AV if received end
	int Protocol(void); /* puts input into mbc_in, and sends from mbc_out */
	};

class ccf_port:public com_port {  // CCF serial communications, either line
	char msg_id[4];  // the message type
	char ack_str[25];  // for ack messages
	int reply(char *,int, msgtype);  // constructs  a reply message
	int prev_seq;	// sequence number of last received packet
public:
	int ack_seq;        // sequence number of ack msg
	ccf_port(unit u1);  // pri or backup line
	ccf_port(const ccf_port &);
	ccf_port(void);
	~ccf_port();
	ccf_port &operator=(const ccf_port &);
	void reset(void);
	int get_seq(void)
		{ return (recbuf[8]+10*recbuf[7]+100*recbuf[6] - 5328);};
	int get_prev_seq(void) { return prev_seq; };
	int CheckEnd(int);	 // Empties the PCCOM receive buffer, sets REC_LINE_AV if received end
	int Protocol(void); // puts input into ccf_in, and sends from ccf_out
	};

#define IM_LEN 14

class weather_port:public com_port { // Weather Monitor serial line, - Primary MSC
	int num_rec;	   // number of bytes to receive
	int send_cmd(void);
	int getwix(int); // get index corresponding to the numerical command,1.04
	void setv(int); // decodes the weatherlink data
	int value[IM_LEN]; // periodic data values
public:
	weather_port(void);
	weather_port(const weather_port &);
	~weather_port();
	weather_port &operator=(const weather_port &);
	void command(int); // formats a command message,index in Wint,param
	};

class sherlock_port:public com_port { // Power Analyzer serial line - Backup MSC
	int rec_msg(int);  // gets one line of the reply, ret 0 on last line
public:
	sherlock_port(void);
	sherlock_port(const sherlock_port &);
	~sherlock_port();
	sherlock_port &operator=(const sherlock_port &);
	void command(int);  // sends a command message and gets response or -1
	};

class parse { // exchanges records between serial ports and the program object
	char *ccf_strng; // makes the CCF out msg
	char *cmd_strng; // gets the CCF commands
	int seq_num;	 // for next packet to send to the CCF lines (both)
public:
	record rdefault; // in case of null pointers
	int get_seq_num(void) { return seq_num;};
	unit role;	 // primary or backup MSC
	rlist ccf_out;   // data to send to the CCF
	rlist mbc_in;	 // messages to the MBC
	rlist mbc_out;   // messages from the MBC
	rlist ccf_in;	 // commands from the CCF
	rlist env_in;    // data from the Weather Monitor/Power Analyzer
	rlist fac_out;   // data from the facility monitoring sensors
	parse(void);
	parse(const parse &); // illegal
	~parse();		 // rlists will get dumped automatically
	parse &operator=(const parse &); // illegal
	int get_param(void);
	int get_ccf(void); /* For the next unprocessed record in ccf.in put it either mbc_out
	or pgm.msc_cntl. Return index in ccf_in if successful, else negative. */

	msgtype put_ccf(void); /* Collect records from rlists mbc_in and loglist plus from records
	msc.com4, msc.env, msc.bit, msc.cst, & msc.status according to priority and add to ccf_out.
	Return and ID of the source or neg if no add. */
	};

// facility control codes
const int PA_PRI_CMD  = 14;
const int PA_BACK_CMD = 13;
const int PA_OFF_CMD  = 12;
const int PA_STBY_CMD = 11;
const int PA_OPER_CMD = 10;
const int POWER_OFF  = 9;
const int POWER_ON   = 8;
const int NUM_DATA    = 12;   // size of data registers in sensor_port
const int D_CHANGED  = 0x08; // bit 3 denotes changed data
const int D_VALID	 = 0x04; // bit 2 denotes valid data

class sensor_port:public port {
protected:
	int g_data[NUM_DATA], n_data[NUM_DATA], c_list[NUM_DATA];  // good and new data, change flag
	sensor s_list[NUM_DATA];		// corresponding sensor identificaton
public:
	sensor_port(unit);
	~sensor_port(void);
	int virtual Update(void) = 0; // each derived class implements differently
	friend ostream &operator<<(ostream &, const sensor_port &); // outputs data
	void display_data(int,int); // writes one value
	};

class lpt_port:public sensor_port {  // Primary MSC port expander object
	int LptGet(const sensor); // gets one sensor value or -1
public:
	lpt_port(void);
	~lpt_port(void);
	int Update(void); // gets data into g_list
	int Cntl(int);	// performs facility control action
	int Check_PA(void); // turns off PA on Utility Power fail
	};

#ifdef TRACE
static const int NUM_SAVED = 17;

class tr_port:public port {
	char digit[NUM_SAVED];
        int k;
	void portprint(char s, char d);  // s = location, d = digit
public:
        tr_port(int p); // port number
        ~tr_port();
	void displaydigit(int num); // number to display next
	void displayled(int which); // led to change display, on <> off
	void display_data(void);
	friend ostream &operator<<(ostream &, tr_port &);
        };
#endif

class pio_port:public sensor_port {   // Backup MSC PIO-24 Digital I/O board object
	void setup(void);
	// PA=in,PB=out,PClow=in,PChigh=out
	int PioGet(const sensor); // gets sensor value or -1, includes reboot
public:
	pio_port(void);
	~pio_port(void);
	int Update(void); // gets data into g_list
	int Reboot(unit); // resets the specified unit(returns immediately)
	void setPAstate(void); // reads the PA relay values into status
        int Check_PA(void); // checks for Standby relay open
	};

class das_port:public sensor_port {  // Backup MSC DAS-4 Analog Input board object
	int GetAnalog(int); // valid return values 0 - 255
	int DasGet(const sensor); // gets the sensor value
public:
	das_port(void);
	~das_port(void);
	int Update(void); // gets data into g_list
	int Cntl(int);	// performs facility control
	int Check_PA(void); // checks for excessive Reflected Power
	};

class pgm {		 // Controls the MSC and the Master Station
	int stat_change;
	int old_core;     // Kb of heap avail
        int num_bigs;     // number of max. length msg buffers available
	struct dfree dtable; // % of disk C avail, filled by test
	ofstream recd;
public:
	int get_disk_free(void) const  {  // depends on test()
		if(dtable.df_total < 100) return -1;
		return (int)(dtable.df_avail/(dtable.df_total/100)); };
	char *diskbuf; // shared for all disk operations
	record bit;	 // Most recent self-test data
	record cst;	 // Most recent communications statistics
	pgm(void);	  // initialization determines local and status
	~pgm();		 // close up program here
	pgm(const pgm &);
	pgm &operator=(const pgm &);
	void test(void); // returns available memory in Kb or -1
	void new_date(int);  // sets system date
	void new_time(int);  // sets system time
	friend ostream &operator<<(ostream &, const pgm &); // outputs data
	void display_stats(void); // displays the memory, etc
	void store_date(void); // stores date,time,endprog in CRASHREC.RST
	};

typedef struct { // global variables settable from CCF or local operator
	char   c[5];   // same as first 4 letters of Cint.c
	unsigned  v;   // value of the variable
	void save_param(void); // puts a command into ccf_in
	} param_t;

typedef struct {      // local operator commands
	char    c[10];  // command
	msgtype     t;  // type for command string to ccf_in
	int         n;  // base number to add to input, or neg if no input
	int         d;  // divisor between command parameter and value
	} cmd_t;

const int NUM_CMD = 19;

const static cmd_t Cint[] = {
{ "active"   , ms_cmd  , -11  , 1}, // sets MBC interface active
{ "backpa"   , ms_cmd  , -2   , 1}, // selects backup PA
{ "bitperiod", envdata , 0    ,10}, // sets CheckTime
{ "clearcomm", com_stat, -9999, 1}, // clears communications stat counts
{ "date"     , pgm_ctl , 5600 , 1}, // ccf sends 2 commands, year & julian
{ "event"    , bitedata, 0    , 1}, // sets debug_level
{ "hitemp"   , ms_cmd  , 0    , 1}, // high int. temp, 75 - 120 deg. F
{ "inactive" , ms_cmd  , -10  , 1}, // sets MBC interface inactive
{ "limit"    , ms_cmd  , 5000 , 1}, // no min pa test if lower than this
{ "pamin"    , ms_cmd  , 5000 , 1}, // min pa output
{ "paoff"    , ms_cmd  , -3   , 1}, // turns PA main relay off
{ "paon"     , ms_cmd  , -5   , 1}, // turns PA main and standby relays on
{ "pastby"   , ms_cmd  , -4   , 1}, // turns PA standby relay off
{ "pgmperiod", pgm_ctl , 0    ,10}, // sets PgmPeriod
{ "pripa"    , ms_cmd  , -1   , 1}, // selects Primary PA (coax relay off)
{ "quit"     , pgm_ctl , 9990 , 1}, // returns to DOS
{ "ratio"    , ms_cmd  , 4000 , 1}, // incident/reflected ratio min
{ "time"     , pgm_ctl , 3200 , 1}, // hh:mm:ss
{ "vswr"     , ms_cmd  , 4000 , 1},}; // max ref. watts

int getcmdix(char *);  // gets index in Cint or -1

class page {     // methods for operator display
    char *cmd_line;
    int cmd_indx;
    rlist &get_page(void);  // gets ref to page given in local_op_mode[0]
	void command_help(void);
    void display_help(void);
    void display_stats(void);
public:
    char *dispbuf;  // for displaying archives
	rlist email;  // to CCF
    rlist msgs;   // from CCF
    page(void);
    ~page();
    void edit_line(int);   // function key is +256
	void new_display(int); // called for a function key
    void update_cmd(int);  // called for anything else
    friend void marker(int); // telltale for program activity
    };

#ifdef TEST_MODE
#include "TEST.H"
#endif

#endif
