// PROGRAM.CPP 1.23 Methods for Environmental Sensing, Facility Control and Self-Test
/* Added display of Interior Temperature to sensor_port::display_data, changed
   line voltage width to 3 digits from 4. Added coefficient INTEMP_CONV. Added check of
   excessive temperature to das_port::CheckPa(), & HIGH_TEMP to its status word  */
// 1.03 global params type change & initialization moved to globals
// 1.04 add CRASHREC.RST file
// 1.1 use clog in log()
// 1.2 TRACE added

#define SCOPE extern
#include "globals.h"
#include <alloc.h>
#include <bios.h>
#include <math.h>

const int TEST_HI = 197;   // 2.5 V +- 4%
const int TEST_LO = 187;
const int REF_CONV = 6; // was 7
const int FWD_CONV = 60;  // was 55
const int LINE_CONV = 1;
const int ADCWait = 5;	 // ms for DAS -4 ADC to get result

static HANDLE cntl_mutex;
static HANDLE cntl_timer;
static HANDLE rebt_timer;
static HANDLE das_timer;
static HANDLE WthrTimer;
static HANDLE Wthr_Event;
static HANDLE das_event;

int *CNTL_STK;  // stack for thread for facility control
int *REBT_STK;  // stack for thread for rebooting
int *W_STK; // Weather Monitor/Power Analyzer Stack
const int P_STK_LEN = 1025;  // pgm_Cntl stack size; 1.03 was variable

void far rebt_responder(void);
void far das_responder(void);
void far cntl_responder(void);
void far wthr_responder(void);
void far PgmResponder(void);

THREAD cntl_thread(void);

// states of the (Selected) Power Amplifier
const int PA_IS_OFF = 0;
const int PA_IS_STBY = 0x200;
const int PA_IS_ON = 0x400;
const int PASTATE = 0x600;   // mask for above
const int PACHANGE = 0x800;
const int POWER_LOSS = 0x4000;
const int HIGH_TEMP = 0x1000; // DAS-4 status word

void atexit_func(void) {
	}

int getix(const unit u1) {
	for(int i = 0; i < NUM_UNIT ; i++) {
		if( Uint[i].u == u1) break;
		}
	if(i == NUM_UNIT) return -1;
	else return i;
	}

int getix(const msgtype t1) {
	for(int i = 0; i < NUM_TYPE ; i++) {
		if( Tint[i].t == t1) break;
		}
	if(i == NUM_TYPE) return -1;
	else return i;
	}

int getix(const sensor s1) {
	for(int i = 0; i < NUM_SENS ; i++) {
		if( Sint[i].s == s1) break;
		}
	if(i == NUM_SENS) return -1;
	else return i;
	}

unit GetUnit(char *idcode) {
	for(int j = 0; j < NUM_UNIT ; j++) {
		if(strncmpi(Uint[j].i,idcode,2) == 0) break;
		}
	if(j == NUM_UNIT) return invunit;
	else return Uint[j].u;
	}

int GetUnitName(char *s1,const unit u1) {
	ostrstream incpy(s1,13);
	incpy << Uint[getix(u1)].a;
	return 12;
	}

int GetUnitCode(char *s1,const unit u1) {
	ostrstream incpy(s1,3);
	incpy << Uint[getix(u1)].i;
	return 2;
	}

msgtype GetType(char *idcode) {
	for(int j = 0; j < NUM_TYPE ; j++) {
		if(strncmpi(Tint[j].i,idcode,3) == 0) break;
		}
	if(j == NUM_TYPE) return invtype;
	else return Tint[j].t;
	}

int GetTypeName(char *s1,const msgtype t1) {
	ostrstream incpy(s1,13);
	incpy << Tint[getix(t1)].a;
	return 12;
	}

int GetTypeCode(char *s1,const msgtype t1) {
	ostrstream incpy(s1,4);
	incpy << Tint[getix(t1)].i;
	return 3;
	}

istream &operator>>(istream &strm, msgtype &mt) {
	char idcode[15];
	strm.get(idcode,15,STX); // scan until STX found (not extracted)
	mt = GetType(idcode + strlen(idcode) - 3);
	return strm;
	}

sensor GetSensor(char *idcode) {
	for(int j = 0; j < NUM_SENS ; j++) {
		if(strncmpi(Sint[j].i,idcode,2) == 0) break;
		}
	if(j == NUM_SENS) return invsens;
	else return Sint[j].s;
	}

int GetSensorName(char *s1,const sensor t1) {
	ostrstream incpy(s1,16);
	incpy << Sint[getix(t1)].a;
	return 15;
	}

int GetSensorCode(char *s1,const sensor t1) {
	ostrstream incpy(s1,3);
	incpy << Sint[getix(t1)].i;
	return 2;
	}

ostream &operator<<(ostream &str, unit u1) {
	int index = getix(u1);
	str << Uint[index].i;
	return str;
	}

ostream &operator<<(ostream &str, msgtype t1) {
	int index = getix(t1);
	str << Tint[index].i;
	return str;
	}

ostream &operator<<(ostream &str, sensor s1) {
	int index = getix(s1);
	str << Sint[index].i;
	return str;
	}

int getix(const wcmd w1) { // index corresponding to the commend enum
	for(int i = 0; i < NUM_WTHR ; i++) {
		if( Wint[i].w == w1) break;
		}
	if(i == NUM_WTHR) return -1;
	else return i;
	}

int weather_port::getwix(int num) { // get index corresponding to the command number,1.04
	for(int i = 0; i < NUM_WTHR ; i++) {
		if(num >= Wint[i].c && num < Wint[i].c + Wint[i].m) break;
		}
	if(i == NUM_WTHR) return -1;
	else return i;
	}

ostream &operator<<(ostream &str, wcmd w1) {
	int index = getix(w1);
	str << Wint[index].i;
	return str;
	}

/*  level guidelines:
	1	Program cannot proceed
	2	Allocation faults
	3	Serious com, hard disk, equipment problems,
	4	Command feedback
	5	Significant routine startup messages
	6	Routine com errors
	7	Detail control actions
	8	Routine program exit messages */

void log(unit u1, const char *s1, int i1, int level) {
	if((debug_level.v & 0x0f) >= level && u1 != logger) {
		ostrstream sout; // declares an output stream
		sout << u1 << ':' << s1;
		if(i1 > 9 || i1 < -3) sout << "(0x" << hex << i1 << ')' << ends;
		else sout << '(' << i1 << ')' << ends;
		char *outstrng = fixstream(sout);
		if(outstrng) {
			loglist.add_rec(outstrng);
			if(local_op_mode[DISP_MODE] == LOG_PAGE && InUse[0] == 0) { // 1.04
				time_t now = time(NULL);
				cerr << outstrng << '@' << ctime(&now);
				}
			if(PRINTLOG > 1) lod.msgs.add_rec(outstrng);
			delete outstrng;
			outstrng = 0;
			}
		else cerr << "Failure to log" << endl;
		}
	}

void log(msgtype t1, char *s1, int i1, int level) {
	if((debug_level.v & 0x0f) >= level && t1 != log_msg) {
		ostrstream sout; // declares an output string
		sout << t1 << ':' << s1 << '(' << i1 << ')' << ends;
		char *outstrng = fixstream(sout);
		if(outstrng) {
			loglist.add_rec(outstrng);
			if(local_op_mode[DISP_MODE] == LOG_PAGE && InUse[0] == 0) { // 1.04
				time_t now = time(NULL);
				cerr << outstrng << '@' << ctime(&now);
				}
	                if(PRINTLOG > 1) lod.msgs.add_rec(outstrng);
			delete outstrng;
		outstrng = 0;
			}
		else cerr << "Failure to log" << endl;
		}
	}

static int cntl_wait;

void far cntl_responder(void) {
	cntl_wait=0;
	}

port::port(const unit u1) {
	u = u1;
	int index = getix(u);
	ident = Uint[index].id;
	addr = Uint[index].addr;
	if(addr > 0) log(u,"Port Address",addr,5); // 1.02 was 6
	status = 0;
	}

port::~port() {
	log(u,"Closing port Status",status,8);
	}

void port::setpstatus(int value, int mask = 0x8000) {
//	EnterCriticalSection();
	status &= ~mask;
	status |= value;
//	LeaveCriticalSection();
	}

void sensor_port::sensor_port(unit u1):port(u1) {
	for(int i=0; i < NUM_DATA; i++) {
		s_list[i] = invsens;
		g_data[i] = 259;
		}
	}

void sensor_port::~sensor_port(void) {
	}

ostream &operator<<(ostream &str, const sensor_port &p1) {
	str << NL << p1.u << SEP << hex;  // 1.21  port ident
	str.width(4); // 2 hex digits for status
	str.fill('0');  // leading zeroes
	str  << p1.status;  // status in hex
	for(int i = 0; i < ((p1.status & 0xf0) >> 4); i++) {
		if(p1.status & 0x07 != 0x07) break;	// no valid data
		int index = getix(p1.s_list[i]);           // 1.24, instead of c_list
	        if(index < 0) continue;
		str << SEP << p1.s_list[i] << dec;
		unsigned char selcode = Sint[index].KPort;
		if(selcode == 3) {
			str.width(3); // analogs get 3 digits
			str << p1.g_data[i];
			}
		else {
			str.width(1);  // one digit for change discrete
			str << p1.g_data[i] << '|' << p1.c_list[i];
			}
		}  // ident string, data, change flag for discretes only
	return str;
	}

void sensor_port::display_data(int i,int s_indx) {
	if(local_op_mode[DISP_MODE] != STAT_PAGE) return;
	AcquireMutex(bios_mutex);
	gotoxy(Sint[s_indx].col,Sint[s_indx].row);
	switch(Sint[s_indx].LptSelect) {  // flag for display mode
		case 3: // analog value
		switch(s_list[i]) {
		    case fwd_pwr: cprintf("%4d/%4u/%4u",
			(g_data[i]-128)*FWD_CONV,
			pa_lower_limit.v,min_inc_pwr.v);break;
		    case ref_pwr: cprintf("%3d/%4u/%u",
			(g_data[i]-128)*REF_CONV,
			max_ref_pwr.v,max_vswr.v);break; // 1.03
		    case level: cprintf("%4d",g_data[i]);break;
		    case latch: cprintf("%4d",g_data[i]);break;
		    case unreg: cprintf("%3d",(g_data[i]-128)*LINE_CONV);break;
		    case intemp: cprintf("%3d",(g_data[i]-INTEMP_CONV));break;
		    case ADC_mon: cprintf("%4d",(g_data[i]-128)*78);break;
		    case fv_supply:cprintf("%4d",(g_data[i]-128)*39);break;
		    } break;
		case 0:case 1:case 5: // safety sensor, display only if not normal
			int t_attr = 0; // black on black
			if(g_data[i]) {
				t_attr = 15;  // bright white
				if(c_list[i]) t_attr = 0xf0; // invert
				}
			textattr(t_attr);
			cputs(Sint[s_indx].a);
			textattr(15); break;
		case 4: if(g_data[i]) cputs("Backup PA is ");
	 		else cputs("Primary PA is"); break;
 		}
	ReleaseMutex(bios_mutex);
	if(local_op_mode[NEW_PAGE]) local_op_mode[NEW_PAGE]++;
	if(local_op_mode[NEW_PAGE] > NUM_DISP) local_op_mode[NEW_PAGE] = 0;
	}

void lpt_port::lpt_port(void):sensor_port(printer) {
	unsigned lptaddr = 8 + 2 * (ident - 1);
	addr = peek(0x40,lptaddr);	 // read the actual bios value
	log(u,"Address of Expander Port",addr,5);
	if(addr == 0x2BC || addr == 0x278 || addr == 0x378) {
		status = 1;
	// test for jumper between bit 6 and the fault/error line, bit 3
		unsigned bit6;	 // value of bit 6
		for(bit6 = 0; bit6 < 2; bit6++) {
			outportb(addr,bit6 * 0x40 + 0x80);   // set the high bit as well
			unsigned char st = (inportb(addr+1) & 0x08) / 8;
			if(bit6 == st) status++;
			}
		}
	switch(status)	{
		case 0: log(u,"This printer port not present:",ident,3);break;
		case 1:	log(u,"Parallel Printer Port Present",status,5);break;
		case 3:	{
		     status |= PA_IS_ON; // assume worst case
		     log(u,"Expander port (STA U #1) present",status,5);break;
#ifdef TRACE
		     trp.setpstatus(0); // don't send debug info to it
#endif
		     }
		default: log(u,"No STA-U #1 installed",status,3);
			status = 0;
		}
	}

void lpt_port::~lpt_port(void) {
	}

int lpt_port::LptGet(const sensor s2) {
	if((status & 0x03) != 3) return 2;
	int index = getix(s2);
	if(index < 0) return 259;
	unsigned char ret;
	unsigned char selcode = Sint[index].LptSelect;
	if(selcode == 0 || selcode == 1) {  // valid for lpt discretes
		AcquireMutex(bios_mutex);
		outportb(addr,selcode * 0x80);  // set up multiplexer
		ret = inportb(addr+1); // get the status bits of lpt
		ReleaseMutex(bios_mutex);
		ret >>= Sint[index].LptBit;  // shift the selected bit into LSB
		ret ^= Sint[index].LptInv;   // invert as necessory
		ret &= 1;					// isolate LSB
		}
	else ret = 2;
	return (int)ret;
	}

void execute(unit, int);

void execute(unit u, int action) {
	unsigned adr;
	unsigned char c = (unsigned char)action;
	AcquireMutex(bios_mutex);
	if(u == printer) {
		adr = expd.getpaddr();
		outportb(adr,c);
		}
	if(u == analog) {
		adr = das.getpaddr();
		outportb(adr+2,c * 0x10);
		}
	ReleaseMutex(bios_mutex);
	log(u,"Detail Control Action",(15 - action),7);
	InUse[3] = 16 - action;
	}

static int statcode;
static int old_code = 0;


THREAD cntl_thread(void) {
	int code = statcode;
	unit board = printer;
	AcquireMutex(cntl_mutex);
	int state = PA_IS_ON;
	if(code > 0 && code < 8) code = 15 - code;
	if(code > 16 && code < 31) {
		board = analog;
		state = pio.getpstatus(PASTATE);
		code -= 16;
		}
	else {  // Primary MSC
		if(code == PA_OPER_CMD | expd.getpstatus(POWER_LOSS) == 0)
			state = PA_IS_OFF;
		}
	if(code == POWER_ON || code == POWER_OFF) {
		if(code == old_code) {
			execute(board,code);
			cntl_wait = 1;
			StartTimer(cntl_timer,60); // min pulse width
			old_code = 0;
			}
		else {
			log(board,"First Disconnect Command registered",code,4);
			old_code = code;
			}
		}
	else {
		code |= state;
		switch(code) {

			case (PA_PRI_CMD | PA_IS_ON): case (PA_BACK_CMD | PA_IS_ON):
				if(com2.get_active() == 0) {
					com2.set_active(2);
					}
					exch.mbc_out.add_rec("IB2b900000,STOP");
				StartTimer(cntl_timer,2000); // wait 2 sec
				cntl_wait = 1;
				while(cntl_wait) PassTimeSlice();

			case (PA_PRI_CMD | PA_IS_STBY): case (PA_BACK_CMD | PA_IS_STBY):
				execute(board,PA_STBY_CMD);  // turn off output
				StartTimer(cntl_timer,5000); // cool down time
				cntl_wait = 1;
				while(cntl_wait) PassTimeSlice();
				execute(board,PA_OFF_CMD);  // then switch off
				StartTimer(cntl_timer,500); // wait a bit
				cntl_wait = 1;
				while(cntl_wait) PassTimeSlice();

			case (PA_PRI_CMD | PA_IS_OFF): case (PA_BACK_CMD | PA_IS_OFF):
				execute(board, (code & 0x0f));
				state = PA_IS_OFF;
				break;

			case (PA_OFF_CMD | PA_IS_ON):
				if(com2.get_active() == 0) {
					com2.set_active(2);
					}
				exch.mbc_out.add_rec("IB2b900000,STOP");
				StartTimer(cntl_timer,2000);
				cntl_wait = 1;
				while(cntl_wait) PassTimeSlice();
				execute(board,PA_STBY_CMD);
				StartTimer(cntl_timer,5000); // cool down time
				cntl_wait = 1;
				while(cntl_wait) PassTimeSlice();

			case (PA_OFF_CMD | PA_IS_STBY):
			case (PA_OFF_CMD | PA_IS_OFF):
				execute(board,PA_OFF_CMD);
				state = PA_IS_OFF;
				break;

			case (PA_STBY_CMD | PA_IS_ON):
				if(com2.get_active() == 0) {
					com2.set_active(2);
					}
				exch.mbc_out.add_rec("IB2b900000,STOP");
				StartTimer(cntl_timer,2000);
				cntl_wait = 1;
				while(cntl_wait) PassTimeSlice();

			case (PA_STBY_CMD | PA_IS_OFF):
			case (PA_STBY_CMD | PA_IS_STBY):
				execute(board,PA_STBY_CMD);
				state = PA_IS_STBY;
				break;

			case (PA_OPER_CMD | PA_IS_OFF):
				execute(board,PA_STBY_CMD);
				StartTimer(cntl_timer,5000); // 11-22-93
				cntl_wait = 1;
				while(cntl_wait) PassTimeSlice();

			case (PA_OPER_CMD | PA_IS_STBY):
			case (PA_OPER_CMD | PA_IS_ON):
				execute(board,PA_OPER_CMD);
				StartTimer(cntl_timer,1500);
				cntl_wait = 1;
				while(cntl_wait) PassTimeSlice();
				if(com2.get_active() == 0) {
					com2.set_active(2);
					}
				exch.mbc_out.add_rec("IB2b900000,START");
				state = PA_IS_ON;
				break;

			default: log(board,"Illegal Power Amplifier Command:",code,3);
			}
		}
	StartTimer(cntl_timer,300);
	cntl_wait = 1;
	while(cntl_wait) PassTimeSlice(); // wait for spike filter
	execute(board,15); // set to rest value
	StartTimer(cntl_timer,2000); // min wait between control actions
	cntl_wait = 1;
	while(cntl_wait) PassTimeSlice();
	if(board == analog) {
		pio.setpstatus(state,(PASTATE | TX_BUF_BUSY));
		das.setpstatus(state,(PASTATE | TX_BUF_BUSY)); // also
		}
	else {
		expd.setpstatus(state,(PASTATE | TX_BUF_BUSY));
		}
	InUse[3] = 0;
	ReleaseMutex(cntl_mutex);
	DeallocateThread();
	}

int lpt_port::Cntl(int c) {
	if(c > 0 && c < 8 && !endprog) {
		statcode = c;
		cntl_handle = AllocateThreadLong(cntl_thread,FP_SEG(CNTL_STK),NPRI);
		status |= TX_BUF_BUSY;
		InUse[3] = 9;
		return 0;
		}
	return -1; // code is invalid
	}

int lpt_port::Check_PA(void) {
    int inv_indx = -1; // location in s_list of inverter on ID
    int num_values = (status & 0x1f0)/16; // number of valid readings
    for(int indx = 0; indx < num_values;indx++) {
		if(s_list[indx] == bypass) break;
		}
    inv_indx = indx;
    static int count = 0;
    if(inv_indx < num_values && inv_indx >= 0) {
		if(g_data[inv_indx] == 1) {
			if(count*PgmPeriod.v > 3000) {
				if(status & POWER_LOSS) {
					q_cntl.push(3); // set PA off
					log(u,"Utility Power Loss",count,3);
					status |= POWER_LOSS;
					}
				count = 0;
				}
			else count++;
			}
		else {
			count = 0;
			if(status & POWER_LOSS) {
				q_cntl.clear();
				log(u,"Utility Power Restored",1,3);
				status &= ~POWER_LOSS;
				}
			}
		}
	return count;
	}

int lpt_port::Update(void) {
	if((status & 0x03) == 3) { // port is working
		int indx = 0;
		status &= ~D_CHANGED;
		status &= ~D_VALID;
		for(int i = 0; i < NUM_SENS; i++) {
			if(Sint[i].KPort == 0) {  // we have an applicable sensor
				int val = LptGet(Sint[i].s);
				if(val == 0 || val == 1) {
					status |= D_VALID;
					n_data[indx] = val;
					s_list[indx] = Sint[i].s;
					if(n_data[indx] != g_data[indx] ) {
//						EnterCriticalSection(); // ensure reading new data
						g_data[indx] = n_data[indx];
						status |= D_CHANGED;
						c_list[indx] = 1;
//						LeaveCriticalSection();
//						local_op_mode[NEW_DATA] = 1;
						}
					else c_list[indx] = 0;
					if(c_list[indx] || local_op_mode[NEW_PAGE])
						display_data(indx,i);
					indx++;
					if(indx >= NUM_DATA) break; // no space for any more
					}
				}
			}
		status &= ~(0x1f0);
		status |= (16 * indx);
		}
	return status;
	}

void pio_port::pio_port(void):sensor_port(digital) {
// test the connection between PB4 and PA7
	status = 1;
	setup();
	unsigned char tval;
	for(unsigned char test = 0; test < 2; test++) {
		tval = (test + 12) * 0x10;   // also turns on the MBC and primary MSC
#ifdef KEEP_ALIVE
		tval += KEEP_ALIVE;
#endif
		outportb(addr+1,tval);  // set PCB to test
		unsigned char st=inportb(addr);  // get value of PA
		if((st & 0x80) == test * 0x80) status ++;
		}
	if(status == 3) {
		tval = 0xc0;  // turn on both IPMs
#ifdef KEEP_ALIVE
		status |= 0x1000; // keep_alive location
		tval += KEEP_ALIVE;
#endif
		outportb(addr + 1,tval); // turn on IPCs and keep_alive
		log(u,"PIO-24 port connected to STAU",status,3);
		}
	else
		log(u,"PIO-24 board not connected to STA-U #2",status,3);
	}

void pio_port::~pio_port(void) {
	}

void pio_port::setup(void) {
	AcquireMutex(bios_mutex);
	outportb(addr + 3,0x91);
	ReleaseMutex(bios_mutex);
	}

int pio_port::PioGet(const sensor s2) {
	if((status & 0x03) != 3) return 2;
	int index = getix(s2);
        if(index < 0) return 259;
	unsigned char selcode = Sint[index].KPort;
	unsigned char ret;
	if(selcode == 0 || selcode == 2) {  // else not on PIO-24
		setup();  // make sure control is set
		AcquireMutex(bios_mutex);
		ret = inportb(addr+selcode); // get the input data
		ReleaseMutex(bios_mutex);
		ret >>= Sint[index].KBit;  // shift the selected bit into LSB
		ret ^= Sint[index].KInv;   // invert as necessory
		ret &= 1;				  // isolate LSB
		}
	else ret = 2;
	return (int)ret;
	}

static int rebt_wait;

void far rebt_responder(void) {
	 rebt_wait=0;
	 }

THREAD rebt_thread(void);

THREAD rebt_thread(void) {
	int offcode = rebt_wait;
	outportb(pio.getpaddr() + 1,offcode);
	StartTimer(rebt_timer,5000);  // wait 5 seconds
	while(rebt_wait) PassTimeSlice();
	InUse[4] = 2;
	outportb(pio.getpaddr() + 1,0xe0);  // back on
	StartTimer(rebt_timer,12000);  // wait 12 seconds
	rebt_wait = 1;
	while(rebt_wait) PassTimeSlice();
	InUse[4] = 0;
	DeallocateThread();
	}

int pio_port::Reboot(unit u) {
	if((status & 0x03) != 3) return 257;
	rebt_wait = 0xc0;
	if(u == mbc) rebt_wait = 0x80;
	if(u == pri) rebt_wait = 0x40;
	if(InUse[4]) {
		if(rebt_wait != 0xc0)
			log(u,"Reboot already in progress,command cancelled",rebt_wait,4);
		return rebt_wait;
		}
	if(rebt_wait == 0xc0) { // do a keep-alive
		int keep_code = getpstatus(0x1000) ^ 0x1000;
#ifdef KEEP_ALIVE
		setpstatus(keep_code,0x1000); // output the bit
		keep_code /= 0x1000/KEEP_ALIVE;
		outportb(addr+1,(keep_code | 0xc0));
#endif
		return keep_code;
		}
	else {
		log(u,"Reboot Command",rebt_wait,4);
		rebt_handle = AllocateThreadLong(rebt_thread,FP_SEG(REBT_STK),NPRI);
		InUse[4] = 1;
		return -1;
		}
	}

int pio_port::Update(void) {
	if((status & 0x03) == 3) { // port is working
		int indx = 0;
		status &= ~(D_CHANGED | D_VALID);
		for(int i = 0; i < NUM_SENS; i++) {
			if(Sint[i].KPort == 0 || Sint[i].KPort == 2) {  // A or C(low)
				int val = PioGet(Sint[i].s);
				if(val == 0 || val == 1) {
					status |= D_VALID;
					n_data[indx] = val;
					s_list[indx] = Sint[i].s;
					if(n_data[indx] != g_data[indx]) {
						g_data[indx] = n_data[indx];
						status |= D_CHANGED;
						c_list[indx] = 1;
						if(Sint[i].KPort == 2) status |= PACHANGE;
						}
					else c_list[indx] = 0;
					if(c_list[indx] || local_op_mode[NEW_PAGE])
						display_data(indx,i);
					indx++;
					if(indx >= NUM_DATA) break;
					}
				}
			}
		status &= ~(0x01f0);
		status |= (indx * 16);
		}
	return status;
	}

int pio_port::Check_PA(void) {
	setPAstate(); // get relay values and put into status word
	if((status & PACHANGE || local_op_mode[NEW_PAGE]) &&
		local_op_mode[DISP_MODE] == STAT_PAGE ) {
		AcquireMutex(bios_mutex);
		gotoxy(54,3);
		clreol();
		switch(status & PASTATE) {
			case PA_IS_ON: cputs("on");break;
			case PA_IS_STBY: cputs("standby");break;
			case PA_IS_OFF: cputs("off");break;
			default: cputs("invalid");
			}
		ReleaseMutex(bios_mutex);
		status &= ~PACHANGE;
		}
	int inv_indx = -1; // location in s_list of inverter on ID
	int num_values = (status & 0x1f0)/16; // number of valid readings
	static int icount = 0;
	for(int indx = 0; indx < num_values;indx++) {
		if(s_list[indx] == bypass) {
			inv_indx = indx;
			break;
			}
		}
	if(inv_indx < num_values && inv_indx >= 0) {
		if(g_data[inv_indx] == 1) {
			if(icount*PgmPeriod.v > 3000) {
				if((status & POWER_LOSS) == 0) {
					log(u,"Utility Power Loss",icount,3);
					q_cntl.push(3); // set PA off
					status |= POWER_LOSS;
					das.setpstatus(POWER_LOSS,POWER_LOSS);
					}
				icount = 0;
				}
			else icount++;
			}
		else {
			icount = 0;
			if(status & POWER_LOSS) {
				q_cntl.clear();
				status &= ~POWER_LOSS;
				log(u,"Utility Power restored",1,3);
				das.setpstatus(0,POWER_LOSS);
				}
			}
		}
	return icount;
	}

void pio_port::setPAstate(void) {
    int state = -1;
    int ps = PioGet(pri_stby);
    int bs = PioGet(back_stby);
    int pm = PioGet(pri_main);
    int bm = PioGet(back_main);
    if((pm == 1 && ps == 0) || (bm == 1 && bs == 0)) state = PA_IS_STBY;
    if((pm == 1 && ps == 1) || (bm == 1 && bs == 1)) state = PA_IS_ON;
    if(pm == 0 && ps == 0 && bm == 0 && bs == 0) state = PA_IS_OFF;
    if(state < 0) log(u,"Invalid Power Amplifier State:",(pm+4*ps+16*bm+64*bs),3);
	else {
		setpstatus(state,PASTATE);
		das.setpstatus(state,PASTATE);
		}
    }

void far das_responder(void) {
         SetEvent(das_event);
	 outportb(das.getpaddr() + 2, 0);  // reset channel and interrupts
	 }

int das_port::GetAnalog(int ch) {
	outportb(addr + 2, ch); // disable interrupts and set mux to ch
	inportb(addr + 1);  // resets A to D, value returned discarded
	outportb(addr,0);		// start the A to D
        ClearEvent(das_event);
	StartTimer(das_timer,ADCWait);  // wait for conversion
	WaitEvent(das_event);
	if(inportb(addr + 2) & 0x80) return 257; // no conversion
	return (int)inportb(addr + 1); // get conversion value
	}

void das_port::das_port(void):sensor_port(analog) {
	 if(AllocateTimer(&das_timer,0,das_responder)) {
		AllocateEvent(&das_event);
		int test = GetAnalog(7);
		if(test > 253 || test < 10) {
			log(u, "No DAS-4 board installed",test,3);
			status = 1;
			return;
			}
		if(test < TEST_HI && test > TEST_LO) {
			status = 3;
			log(u, "DAS-4 Initialized",test,5);
			}
		else {
			log(u, "DAS-4 failed test",test,3);
			status = 2;
			}
		}
	else log(u,"Cannot allocate timer",status,2);
	}

void das_port::~das_port(void) {
	if(status > 0) {
		DeallocateTimer(das_timer);
		DeallocateEvent(das_event);
                }

	}

const static unsigned HD_DELAY = 200; // delay in ms between Vf or Vr measurements
const static int HD_TRIES = 5;  // number of attempts to measure during transmissions
const static int MIN_VALID_POWER = 135;

int das_port::DasGet(const sensor s1) {
	if((status & 0x03) != 3) return 258;
	int index = getix(s1),ret;
        if(index < 0) return 259;
	unsigned char selcode = Sint[index].KPort;
	unsigned char val;
	switch(selcode) {
		case 3:		// analog value
			selcode = Sint[index].KBit;
                        int hd_tries = 1;
                        if(s1 == fwd_pwr || s1 == ref_pwr) hd_tries = HD_TRIES;
			for(int t=0; t < hd_tries; t++) {
				if((ret = GetAnalog(selcode)) > MIN_VALID_POWER) break;
                                ClearEvent(das_event);
				StartTimer(das_timer,HD_DELAY);
                        	WaitEvent(das_event);
                                }
                	break;
		case 4:										 // discrete
			AcquireMutex(bios_mutex);
			val = inportb(addr+2);	 // get the status data
			ReleaseMutex(bios_mutex);
			val >>= Sint[index].KBit;  // shift the selected bit into LSB
			val ^= Sint[index].KInv;   // invert as necessory
			val &= 1;				  // isolate LSB
			ret = (int)val;
			break;
		default: ret = 256;
		}
	return ret;
	}

int das_port::Cntl(int c) {
	if(c > 0 && c < 8 && !endprog) {
		statcode = 31 - c;  // range identifies use of DAS-4 board
		status |= TX_BUF_BUSY;
		cntl_handle = AllocateThreadLong(cntl_thread,FP_SEG(CNTL_STK),NPRI);
		InUse[3] = 10;
		}
	return -1; // code is invalid
	}

int das_port::Update(void) {
	if((status & 0x03) == 3) { // port is working
		int indx = 0;
		status &= ~D_CHANGED;
		status &= ~D_VALID;
		for(int i = 0; i < NUM_SENS; i++) {
			if(Sint[i].KPort == 3 || Sint[i].KPort == 4) {  //Analog,Digital
				int val = DasGet(Sint[i].s);
				c_list[indx] = 0;
				if(val < 256) {
					status |= D_VALID;
					n_data[indx] = val;
					s_list[indx] = Sint[i].s;
					if(n_data[indx] != g_data[indx]) {
						g_data[indx] = n_data[indx];
						c_list[indx] = 1; // used to show changes
						if(Sint[i].KPort == 4) status |= D_CHANGED;
						}
					if(c_list[indx] || local_op_mode[NEW_PAGE])
						display_data(indx,i);
					indx++;
					if(indx >= NUM_DATA) break;
					}
				}
			}
		status &= ~(0x1f0);  // clear bits 4-8
		status |= (indx * 16);
		}
	return status;
	}

int das_port::Check_PA(void) {
	int num_values = (status & 0x1f0)/16; // number of valid readings
	int ref_indx = -1; // location in s_list of reflected power ID
	int inc_indx = -1; // forward power index
	int latch_indx = -1; // location in s_list of reflected power ID
	int temp_indx = -1;
	for(int indx = 0; indx < num_values;indx++) {
		if(s_list[indx] == ref_pwr) ref_indx = indx;
		if(s_list[indx] == fwd_pwr) inc_indx = indx;
		if(s_list[indx] == latch) latch_indx = indx;
		if(s_list[indx] == intemp) temp_indx = indx;
		}
	static int old_vswr = 0;
	static int old_power = 0;
	static int scount = 0;
/*	if(latch_indx < num_values && latch_indx >= 0 &&
		(status & (POWER_LOSS | PASTATE)) == PA_IS_ON) {
		if(g_data[latch_indx] > 192) {
			if(scount*PgmPeriod.v > 3000) {
				q_cntl.push(4); // set PA to Standby
				log(u,"Standby relay open",scount,3);
				scount = 0;
				}
			else scount++;
			}
		else scount = 0;
		}    hardware problem at Boise  */
	if(ref_indx >= 0 && inc_indx >= 0 &&
		(status & (PASTATE | POWER_LOSS)) == PA_IS_ON) {
		int ref_pwr = (g_data[ref_indx] - 128) * REF_CONV;
		int inc_pwr = (g_data[inc_indx] - 128) * FWD_CONV;
		if(ref_pwr > max_ref_pwr.v ||
			(ref_pwr > 20 && inc_pwr/ref_pwr < max_vswr.v)) {
			if(old_vswr*PgmPeriod.v > 3000) {
				q_cntl.push(4); // set PA to Standby
				log(u,"Excessive Reflected Power",ref_pwr,3);
				old_vswr = 0;
				}
			else old_vswr++;
			}
		else old_vswr = 0;
		if(inc_pwr < min_inc_pwr.v && inc_pwr > pa_lower_limit.v) {
			if(old_power*PgmPeriod.v > 3000) {
				q_cntl.push(4); // set PA to Standby
				log(u,"Insufficient Forward Power",inc_pwr,3);
				old_power = 0;
				}
			else old_power++; // needs 3 sec delay
			}
		else old_power = 0;
		}
	static int tcount = 0;
	if(temp_indx < num_values && temp_indx >= 0) {
		if(g_data[temp_indx] - INTEMP_CONV >= max_in_temp.v) {
			if(tcount*PgmPeriod.v > 10000 && (status & HIGH_TEMP) == 0) {
				log(u,"Interior Temperature Limit Exceeded",g_data[temp_indx] - INTEMP_CONV,3);
				q_cntl.push(6); // disconnect power first time
				status |= HIGH_TEMP;
				}
      		if(tcount*PgmPeriod.v > 20000) {
				q_cntl.push(6); // disconnect power
				log(u,"Interior Temperature Disconnect",g_data[temp_indx] - INTEMP_CONV,3);
				tcount = 0;
				}
			}
		if(g_data[temp_indx] - INTEMP_CONV < max_in_temp.v - 5) {
			tcount = 0;
			if(status & HIGH_TEMP) {
				log(u,"Interior Temperature now OK",g_data[temp_indx] - INTEMP_CONV,3);
				status &= ~HIGH_TEMP;
				}
			}
		}
	return old_vswr + old_power + scount + tcount;
	}

#ifdef TRACE
const unsigned STROBE=0x0D;
const unsigned NORMAL=0x0C;

void tr_port::tr_port(int PORTNUM):port(printer) {
	unsigned tr_addr = 8 + 2 * (PORTNUM - 1);
	addr = peek(0x40,tr_addr);	 // read the actual bios value
	if(addr == 0x2BC || addr == 0x278 || addr == 0x378) {
		cerr << "Trace port established at address " << hex << addr << endl;
		status = 3;
		}
	for(int i = 0; i < NUM_SAVED; i ++) digit[i] = 15; // blank
	endprog = 99;
	k = 0;
	}

void tr_port::~tr_port(void) {
	displaydigit(k);
        }

void tr_port::portprint(char s, char d) {  // s = location, d = digit
        if(status) {
		if(endprog < 99) AcquireMutex(bios_mutex);
		outportb(addr,(s<<4)|d);
		outportb(addr+2,STROBE);
		outportb(addr+2,NORMAL);
                if(endprog < 99) ReleaseMutex(bios_mutex);
        	}
	}

void tr_port::displaydigit(int num) { // number to display next
        static int j;
        if(num == 5) k++;
        j = (j == 14) ? 12 : 14;
	for(int i = NUM_SAVED - 4; i >= 0; i--) digit[i+3] = digit[i];
	if(num < 0) {
        	digit[2] = 13; // '-'
                num = -num;
                }
        else digit[2] = j;
	num %= 100;
	digit[0] = (char)(num % 10);
	digit[1] = (char)(num / 10);
	for(i = 0;i < 8; i++) portprint(i,digit[i]);
	}

void tr_port::displayled(int which) { // led to change
        if(which < 0) return; // 0=UTBET,1=FEL,2=BET,3=REDO
	int indx = which % 4;
	const char code[4] = { 8, 9, 12, 13 };
        static char sign[4] = { 0x0f, 0x0f, 0x0f, 0x0f };
        if(sign[indx]) sign[indx] = 0;
	else sign[indx] = 0x0f;
	portprint(code[indx], sign[indx]);
	}

void tr_port::display_data(void) {
	if(local_op_mode[DISP_MODE] != STAT_PAGE) return;
	AcquireMutex(bios_mutex);
	gotoxy(7,21);
	for(int i = 0; i < NUM_USE; i++) cprintf("%d:%d ",i,InUse[i]);
        ReleaseMutex(bios_mutex);
        }

ostream &operator<<(ostream &strm, tr_port &tr) {
        for(int i = NUM_SAVED-1; i >= 0; i--) {
        	char let;
        	if(tr.digit[i] > 9) let = SEP;
        	else  let = tr.digit[i] + '0';
                strm << let;
		}
	return strm;
        }

#endif

pgm::pgm(void) {  // declare msc (sole pgm object) first
	local_op_mode[DISP_MODE] = LOG_PAGE;
	stat_change = 1;
	tzset();
	env_flag = 0;
	b_flag = 0;
	c_flag = 0;
	speed_cnt = 0;
	AllocateMutex(&rlist_mutex);
	AllocateMutex(&bios_mutex);
	endprog = 0;
	AllocateMutex(&cntl_mutex);
	AllocateEvent(&PgmEvent);
	diskbuf = new char[MAX_CHAR + 2];
	cntl_wait = 0;
	rebt_wait = -1;
	CNTL_STK = new int[S_LEN];
	REBT_STK = new int[S_LEN];
	P_STK = new int[P_STK_LEN];
	W_STK = new int[S_LEN];
	if(diskbuf == NULL || CNTL_STK == NULL || REBT_STK == NULL ||
		P_STK == NULL || W_STK == NULL) exit(5);
	for(int i = 0; i < NUM_USE; i++) InUse[i] = 0;
#ifdef TEST_MODE
	for(i=0;i<P_STK_LEN;i++) {
		P_STK[i] = STKMARK;
		}
	for(i=0;i<S_LEN;i++) {
		CNTL_STK[i] = STKMARK;
		REBT_STK[i] = STKMARK;
		W_STK[i] = STKMARK;
		}
#endif
	AllocateTimer(&cntl_timer,0,cntl_responder);
	AllocateTimer(&rebt_timer,0,rebt_responder);
	AllocateTimer(&WthrTimer, 0, wthr_responder);
        AllocateEvent(&Wthr_Event);
	AllocateTimer(&PgmTimer,0,PgmResponder); // executes CCF commands
	AllocateTimer(&PgmWatchDogTimer,0,PgmWatchDogResponder); // may restart program
        struct ffblk f;
        if(findfirst("C:\\OGDEN.ID",&f,0) == 0) {
		MS_ID = 1;
                INTEMP_CONV = 48; // values using Weather Monitor as standard
                }
        else {
		MS_ID = 0;
                INTEMP_CONV = 54;
                }
	}

pgm::~pgm() {
	stat_change = 0;
	StopTimer(PgmTimer);
	AcquireMutex(cntl_mutex);	 // ensures timer not running
	DeallocateTimer(cntl_timer);
	ReleaseMutex(cntl_mutex);
	DeallocateTimer(rebt_timer);
	DeallocateTimer(WthrTimer);
        DeallocateEvent(Wthr_Event);
	DeallocateTimer(PgmWatchDogTimer);
	DeallocateTimer(PgmTimer);
#ifdef TEST_MODE
	AcquireMutex(bios_mutex);
	int i,j;
	j = 0;
	for(i=0;i<S_LEN;i++) if(CNTL_STK[i] != STKMARK) j++;
	cerr << "Facility Control Stack used=" << j << endl;
	j = 0;
	for(i=0;i<S_LEN;i++) if(REBT_STK[i] != STKMARK) j++;
	cerr << "Reboot Stack used=" << j << endl;
	j = 0;
	for(i=0;i<P_STK_LEN;i++) if(P_STK[i] != STKMARK) j++;
	cerr << "Program Control Stack used=" << j << endl;
	j = 0;
	for(i=0;i<S_LEN;i++) if(W_STK[i] != STKMARK) j++;
	cerr << "Weather Monitor Stack used=" << j << endl;
	ReleaseMutex(bios_mutex);
#endif
	delete []P_STK;
	delete []CNTL_STK;
	delete []REBT_STK;
	delete []W_STK;
	DeallocateMutex(rlist_mutex);
	DeallocateMutex(bios_mutex);
	DeallocateMutex(cntl_mutex);
	DeallocateEvent(PgmEvent);
	if(diskbuf) delete []diskbuf;
	diskbuf = NULL;
	}

void pgm::test(void) {	  // returns free memory in Kb
	AcquireMutex(bios_mutex); // 1.02, enabled
	getdfree(3,&dtable);   // get disk C stats and enter into table
	ReleaseMutex(bios_mutex);
	EnterCriticalSection();
	if(farheapcheck() != _HEAPCORRUPT) {
		long core = coreleft();   // space above top allocated block
		num_bigs = core / MAX_CHAR+1;
		struct heapinfo hi;
		hi.ptr = NULL;
		while(farheapwalk(&hi) == _HEAPOK) {
			if(!hi.in_use ) {  // 1.22
				core += hi.size;
				num_bigs += hi.size/(MAX_CHAR + 1);
				}
                        }
		old_core = core/1024;  // 1.02 moved
		}
	else endprog = 4;  // exit, can't go on
	LeaveCriticalSection();
	if(endprog == 4) {
		cerr << "\nHeap is corrupt!\n";
        	disp_use();
		return;
		}
	if(endprog != 4 && num_bigs < MIN_MSGS) { // 1.02 endprog added, core
		log(program,"Low core reboot",num_bigs,1);
		endprog = 3;
		return;
		}
	}

ostream &operator<<(ostream &str, const pgm &p1) {
    str.width(3);
	str << dec << p1.old_core << SEP << p1.num_bigs <<
	SEP << p1.get_disk_free();
    return str;
    }

static struct date reset;
static struct time t;

void rev_julian(int j) {  // sets  da_day, da_mon from julian day
    int leap_year = ((reset.da_year%4) == 0 && reset.da_year != 2000) ? 1 : 0;
    for(reset.da_mon = 1; reset.da_mon < 13; reset.da_mon++) {
		if(j <= (int)day_in_month[reset.da_mon - 1]) break;
	j -= (int)day_in_month[reset.da_mon - 1];
	if(reset.da_mon == 2) j -= leap_year;
	}
    reset.da_day = j;
    }

void rev_mins(int m) {
	t.ti_hour = m / 60;
    t.ti_min = m % 60;
    }


void pgm::new_date(int cp) {
    getdate(&reset); // get current value
    if(cp < 5200) reset.da_year = cp - 3200; // year
    if(cp > 5600 && cp < 5966) rev_julian(cp - 5600);  // julian day
    setdate(&reset);
    log(program,"Date set",cp,4);
	}

void pgm::new_time(int cp) {
	gettime(&t); // get current value
    if(cp < 4640) rev_mins(cp - 3200); // mins past midnight;
    if(cp > 4999 && cp < 5059) t.ti_sec = cp - 5000;
    t.ti_hund = 0;
    settime(&t);
    log(program,"Time set",cp,4);
	}
const int NUM_SPEED = 5;
static int av_speed[NUM_SPEED]; // 1.03 added to see if it decreases

void pgm::display_stats(void) {
//	EnterCriticalSection();
	for(int i = NUM_SPEED - 2; i >=0 ; i--) av_speed[i+1] = av_speed[i];
	av_speed[0] = speed_cnt;
	speed_cnt = 0;
//	LeaveCriticalSection();
	if(local_op_mode[DISP_MODE] == STAT_PAGE) {
		AcquireMutex(bios_mutex);
		gotoxy(76,22);
        	cpr
				 << b_line
				 << wth
				 << pwr  << ends;  // 1.21 was "\r\n"
			char *com_strng = fixstream(cstrm);
			if(cstrm.good()) {
				record *comm_rec;
				comm_rec = new record(1,com_strng,NULL);
				if(comm_rec) {
					AcquireMutex(rlist_mutex);
					msc.cst = *comm_rec;
					ReleaseMutex(rlist_mutex);
					delete comm_rec;
					comm_rec = 0;
					msc.cst.set_flag(N_FLAG);
					}
				}
			delete com_strng;
			com_strng = 0;
#ifdef TRACE
			trp.displaydigit(88);
#endif
			}
		InUse[2] = 6; // 1.03, added
		p_line.display_stats();
		com2.display_stats();
		if(com2.get_active() == 2 && exch.mbc_out.find_rec(N_FLAG) < 0)
			com2.set_active(0); // was temp. active
		b_line.display_stats();
		wth.display_stats();
		pwr.display_stats();
		InUse[2] = 7; // 1.03, added
		if(env_flag) {
			ostrstream envstrm;
			envstrm << expd << pio << das << ends;
			char *envstrg = fixstream(envstrm);
			if(envstrm.good()) {
				exch.fac_out.add_rec(envstrg); // transfer record to rlist
				}
			delete envstrg;
			envstrg = 0;
#ifdef TRACE
			trp.displaydigit(89);
#endif
			env_flag = 0;
			}
		marker(1);
#ifdef TRACE
                trp.displayled(3);
#endif
		}
	InUse[2] = 0;  // finished;1.03 moved
	StopTimer(PgmTimer);
	StopTimer(PgmWatchDogTimer);
	DeallocateThread();
	}
33
Hl/   	GLOBALS.HMSC124// globals.h 1.22 - common external variables for MSC program
// 1.02 - KEEP_ALIVE global
// 1.03 - made settable global variables param_t & initialize
// 1.22 - added MainEvent

// #define TEST_MODE 1
#define KEEP_ALIVE 0x0020;  // Port B, bit 5

#ifndef MASTERH
#include "master.h"
#endif

#ifndef SCOPE
#define SCOPE extern
#endif

// objects must be declared in an order which depends on their constructors
SCOPE int MS_ID;   // 0 = Boise, 1 = Ogden
SCOPE int speed_cnt;         // to measure program activity
SCOPE int b_flag;            // command to perform BIT tests & report
SCOPE int c_flag;            // command to report communications statistics
SCOPE int env_flag;          // command to send a sample of sensor data
SCOPE int endprog;           // program exits if non-zero
/* 0 - running
   1 - local operator exit to DOS
   2 - local operator exit to ARemote
   3 - low memory : pgm::test, restart
   4 - corrupt heap : pgm::test, reboot (& all higher)
   5 - command execution WatchDog timeout
   6 - alloc fail : record::add_str
   7 - alloc fail : record::op =
   8 - alloc fail : record::copy const
   9 - self-test thread WatchDog timeout
   10 - X.25 handler bad
   11 - MBC handler bad
   12 - Async handler bad
   13 - Env. handler bad
   14 - 20 field stream failures
   21 - wild record pointer from get_flag(),get_len(),get_nf()
   22 - attempt to delete null pointer field::~field()
   29 - Com Watch Dog
   34 - PCCOM not running            */
SCOPE int InUse[NUM_USE];    // flag to delay exit while threads busy
SCOPE int local_op_mode[2];  // enables console scrolling, controls display
SCOPE int INTEMP_CONV;  // 1.02 was 50, Boise showed 76 for wth mon=72
SCOPE char *fixstream(ostrstream &); // encapsulates str() function
SCOPE HANDLE rlist_mutex;    // guards multiprocessing operations on rlists
SCOPE HANDLE bios_mutex;     // separates bios operations, non-reentrant
SCOPE HANDLE PgmWatchDogTimer;  // looks for process fail
SCOPE HANDLE PgmTimer;	// Performs control processing periodically
SCOPE HANDLE PgmHandle;   // thread for program control
SCOPE HANDLE CheckerHandle; // thread for self-test
SCOPE HANDLE CheckerEvent; // controls cycling of CheckLists
SCOPE HANDLE Com1Handle;   // thread for X.25 communications
SCOPE HANDLE Com3Handle;   // thread for X.25 communications
SCOPE HANDLE Com2Handle;   // thread for X.25 communications
SCOPE HANDLE cntl_handle;   // thread for PA control
SCOPE HANDLE rebt_handle;   // thread for reboot
SCOPE HANDLE WthHandle;	// thread for Weather Monitor/Power Analyzer
SCOPE HANDLE PgmEvent;    // times the program control thread
SCOPE HANDLE MainEvent;   // times the main loop
SCOPE THREAD pgm_cntl(void);
SCOPE int *P_STK;	 // program control stack
#ifdef MAIN
#ifdef TRACE
tr_port trp(1);         // 8-digit display for debugging
#endif
queue q_cntl(32);      // Q of facility contol commands
queue q_wthr(16);      // Q of commands for the weather monitor/sherlock
#endif
SCOPE pgm msc;    // performs initialization
#ifdef MAIN
param_t max_ref_pwr    = { "vswr",  200 };
param_t max_vswr       = { "rati",    8 };
param_t pa_lower_limit = { "limi",  200 };
param_t min_inc_pwr    = { "pami", 1000 };
param_t max_in_temp    = { "hite",   90 };
param_t debug_level    = { "even",    5 };
param_t PgmPeriod      = { "pgmp",  714 };
param_t MainPeriod     = { "mper",  89 };
param_t CheckTime      = { "bitp",23676 };
rlist loglist(log_msg); // logging file
mbc_port com2(mbc);    // mbc port
ccf_port b_line(backup); // backup line
ccf_port p_line(pri);    // main line
#else
extern param_t max_ref_pwr;       // watts reflected for shutdown to standby
extern param_t max_vswr;          // min ratio of incident to reflected
extern param_t pa_lower_limit;    // test not done if incident is lower
extern param_t min_inc_pwr;       // watts output for shutdown to standby
extern param_t max_in_temp;       // internal temp for shutdown 75-120 deg F
extern param_t debug_level;       // default debug level
extern param_t PgmPeriod;    // ms between command processing
extern param_t MainPeriod;    // ms between keystroke and parse processing
extern param_t CheckTime;    // archive cycle time
extern rlist loglist; // logging file
extern ccf_port p_line;    // main line, double meaning for pri
extern ccf_port b_line; // backup(async) line, alternate meaning for backup
extern mbc_port com2;    // mbc port
extern queue q_cntl;      // Q of facility contol commands
extern queue q_wthr;      // Q of commands for the weather monitor/sherlock
#ifdef TRACE
extern tr_port trp;          // 8-digit display for debugging
#endif
#endif
SCOPE page lod;   // local operator screen and command object
SCOPE lpt_port expd;  // Primary MSC environmental/control port
SCOPE pio_port pio;  // Backup MSC environmental/control port
SCOPE weather_port wth; // Weather Monitor port, Primary MSC
SCOPE sherlock_port pwr; // Power Analyzer port, Backup MSC
SCOPE das_port das;  // A/D port for Power Amp readings, Backup MSC
SCOPE parse exch; // holds most rlists and methods to transfer records
#ifdef SIMULATION
SCOPE int list_flag;         // command to execute CMD_LIST.TXT
SCOPE ccf_sim sim; // simulates CCF commands to MBC for testing if TEST_MODE set
#endif
33
n/   
SERIAL.CPPMSC124// SERIAL.CPP 1.25 Definition of MSC serial interfaces
// 1.02 ccf retry flag decrement
// 1.03 globals use c_param, XOFF on CCF
// 1.22 Transmit changed to reduce processor load during flow control

#define SCOPE extern
#include "globals.h"
#include "..\inc\CCITT.H"

#ifdef TEST_MODE
const int RESEND_FLAG = E_FLAG;
#else
const int RESEND_FLAG = E_FLAG; // was N_flag 5/19

#endif

const int REQUEST_LIMIT = 10; // number of re-requests in a batch

int com_port::IRQset = 0; // this ensures that the IRQ will be set
static int p_timed_out;
HANDLE p_timer;		// MBC - MSC protocol timer
com_port::com_port(unit u1):port(u1) {
	recbuf = new char[MAX_CHAR + 2];
	sendstr = new char[MAX_CHAR + 2];
	if(recbuf == NULL || sendstr == NULL) endprog = 22;
	num_send = MAX_CHAR; // sends as string unless Weather Monitor
	sindx = -1;
	rindx = 0;
	sendstr[0] = '\0';
	recbuf[0] = '\0';
	for(int i = 0; i < 3; i++) msg_cnt[i] = 0;
	stat_change = 1;
	EnterCriticalSection();
	if(cominstalled(COMIRQ)) {  //PCCOM TSR is running
		comsetport(COMIRQ, ident, addr);
		loes on sequence number
			begin << srce << dest << FS << setw(3) << dec << seq_num <<
			FS << sendtype << STX;
			if(srce != ccf && sendtype != mbc_data)
				begin << nowtime << SEP; // not for test or mbc messages
		AcquireMutex(rlist_mutex);
			begin << *source_rec << ends;  // base field method
			ReleaseMutex(rlist_mutex);
			indx = ccf_out.add_rec(ccf_strng);
			if(indx >= 0) {
				ccf_out[indx].set_seqno(seq_num); // set its sequence number for retry
				source_rec->set_flag(D_FLAG); // cancel the source flag
				seq_num++;
				if(seq_num > 999) seq_num = 0;
				}
			}
#ifdef TRACE
		trp.displaydigit(60 + getix(sendtype));
#endif
		}
	return sendtype;
	}
/*
msgtype parse::put_ccf(void) { // Moves packets from multiple places to ccf_out
	int indx = -1;
	msgtype sendtype = invtype;
	if((indx = mbc_in.find_rec(N_FLAG)) >= 0) {
		sendtype = mbc_data;		 // MBD
		}
	if(sendtype == invtype && (indx = lod.email.find_rec(N_FLAG)) >= 0) {
		sendtype = ro_data;		     // ROD
		}
	if(ccf_out.get_nrec() < ((3*ccf_out.get_mrec())/4)) {  // reserve top 1/4, 1.03 was 1/3
		if(sendtype == invtype && (indx = fac_out.find_rec(N_FLAG)) >= 0) {
			sendtype = stat_msg;		  // MSS
			}
		if(sendtype == invtype && (indx = loglist.find_rec(N_FLAG)) >= 0) {
//			if(local_op_mode[DISP_MODE] == LOG_PAGE)  {
//				time_t now = time(NULL);
//				AcquireMutex(bios_mutex);
//				cerr << loglist[indx] << '@' << ctime(&now) << endl;
//				ReleaseMutex(bios_mutex);
//				}
			sendtype = log_msg;		   // LOG
			}
		if(sendtype == invtype && (indx = env_in.find_rec(N_FLAG)) >= 0) {
			sendtype = envdata;		  // EVD
			}
		if(sendtype == invtype && msc.cst.get_flag() == N_FLAG) {
			sendtype = com_stat;		 // CST
			}
		if(sendtype == invtype && msc.bit.get_flag() == N_FLAG) {
			sendtype = bitedata;		 // BTD
			}
		}
	if(ccf_strng && sendtype != invtype) {
		ostrstream begin(ccf_strng,MAX_CHAR);
		if(begin.good()){
			unit dest = ccf;
			unit srce = role;
			ts nowtime; // add the present time in unix code to all ccf msgs
			begin.fill('0');   // leading zeroes on sequence number
			begin << srce << dest << FS << setw(3) << dec << seq_num <<
			FS << sendtype << STX;
			if(srce != ccf && sendtype != mbc_data)
				begin << nowtime << SEP; // not for test or mbc messages
	       		AcquireMutex(rlist_mutex);
                        switch(sendtype) {
                        	case mbc_data: begin << mbc_in[indx];
				    mbc_in[indx].set_flag(D_FLAG);break;
                        	case ro_data:  begin << lod.email[indx];
				    lod.email[indx].set_flag(D_FLAG);break;
                        	case stat_msg: begin << fac_out[indx];
				    fac_out[indx].set_flag(D_FLAG);break;
                        	case log_msg:  begin << loglist[indx];
				    loglist[indx].set_flag(D_FLAG);break;
                        	case envdata:  begin << env_in[indx];
				    env_in[indx].set_flag(D_FLAG);break;
                        	case com_stat: begin << msc.cst;
				    msc.cst.set_flag(D_FLAG);break;
                        	case bitedata: begin << msc.bit;
				    msc.bit.set_flag(D_FLAG);break;
                                default:       begin << rdefault;
                        	}
			begin << ends;  // base field method
			ReleaseMutex(rlist_mutex);
			indx = ccf_out.add_rec(ccf_strng);
			if(indx >= 0) {
				ccf_out[indx].set_seqno(seq_num); // set its sequence number for retry
				seq_num++;
				if(seq_num > 999) seq_num = 0;
				}
			}
#ifdef TRACE
		trp.displaydigit(60 + getix(sendtype));
#endif
		}
	return sendtype;
	}
*/
int parse::get_param(void) {
	int ret=0,j;
	for(int i = 4; i < 8; i++) {
		if((j=cmd_strng[i]) < 48 || j > 57) break;
		ret *= 10;
		ret += j - 48;
		}
	return ((i == 4) ? -1 : ret);
	}

int parse::get_ccf(void) { // analyzes CCF commands
	int indx,done = 1;
	if(InUse[3] || InUse[4] || InUse[5]) return 0;
	if((indx = ccf_in.find_rec(N_FLAG)) < 0) return -1;
	if(ccf_in[indx].getfs(80,cmd_strng) < 1) return -1;  // get the data
	int c_param = get_param();  // all commands have a parameter
	msgtype intype;	   // type of command
	intype = GetType(cmd_strng);
	log(intype,"CCF command:",c_param,7);
	switch(intype) {
		case pgm_ctl:
			if(c_param > 9990 && c_param <= 9999) {
				log(intype,"Abort Program from CCF",4,4);
				endprog = c_param - 9990;
				}
			if(c_param > 29 && c_param < 3200) {
				PgmPeriod.v = 10 * c_param;
				log(intype,"New command processing period",PgmPeriod.v,4);
				}
			if(c_param > 3199 && c_param < 5060) msc.new_time(c_param);
			if(c_param > 5169 && c_param < 5966) msc.new_date(c_param);
#ifdef TRACE
			trp.displaydigit(90);
#endif
			break;

		case bitedata:
			if(c_param > 1 && c_param < 64) {
				debug_level.v = c_param;
				log(intype,"New logging level",debug_level.v,4);
				}
#ifdef TRACE
			trp.displaydigit(91);
#endif
			b_flag = c_param; break; // program status

		case com_stat: c_flag = c_param;
#ifdef TRACE
				trp.displaydigit(92);
#endif
				break; // comm. stats

		case ms_cmd:
			if(c_param > 0 && c_param < 8)
				done = q_cntl.push(c_param); // facility control
			if(c_param == 11) {
				com2.set_active(3);
				log(intype,"MBC Interface now active",c_param,4);
				}
			if(c_param == 10) {
				com2.set_active(0);
				log(intype,"MBC Interface now inactive",c_param,4);
				}
			if(c_param == 12) pio.Reboot(pri);
			if(c_param == 13) pio.Reboot(mbc);
			if(c_param > 74 && c_param < 121) {
				max_in_temp.v = c_param;
				log(intype,"New Internal Temperature Limit:",c_param,4);
				}
			if(c_param > 999 && c_param < 4000) {
            	log(intype,"Sherlock or Weather Monitor Command Received",c_param,4);
				done = q_wthr.push(c_param);
            	}
			// commands to Weather Monitor or Power Analyzer
			if(c_param > 4049 && c_param < 5000) {
				max_ref_pwr.v = c_param - 4000;
				log(intype,"New max ref Limit:",max_ref_pwr.v,4);
				}
			if(c_param > 5399 && c_param < 8000) {
				min_inc_pwr.v = c_param - 5000;
				log(intype,"New Minimum PA Power:",min_inc_pwr.v,4);
				}
			if(c_param > 3999 && c_param < 4050) {
				max_vswr.v = c_param - 4000;
				log(intype,"New VSWR Limit:",max_vswr.v,4);
				}
			if(c_param > 4999 && c_param < 5400) {
				pa_lower_limit.v = c_param - 5000;
				log(intype,"New PA Power lower limit",pa_lower_limit.v,4);
				}
#ifdef TRACE
			trp.displaydigit(93);
#endif
			env_flag = 1;  // send facility monitoring data
			break;

		case envdata:
			if(c_param > 99 && c_param < 3200) {
				CheckTime.v = 10 * c_param;
				log(intype,"New Self-test interval",CheckTime.v,4);
				}
#ifdef TRACE
			trp.displaydigit(94);
#endif
			env_flag = 1; break;

		default: log(intype,"Invalid CCF message",indx,3);
		}
	if(done >= 0) ccf_in[indx].set_flag(D_FLAG);  // record is done - cancel
	return indx;
	}

parse::~parse() {
/*	int ret,left;
	ret=ccf_out.find_rec(N_FLAG);
	left = (ret < 0) ? 0 : ccf_out.get_nrec() - ret;
	log(ccf,"CCF records unsent",left,3);
	ret=mbc_out.find_rec(N_FLAG);
	left = (ret < 0) ? 